# coding=utf-8
import logging
import os
import shutil

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.projects.mt.product import TranslateWebLxcContainer
from sandbox.projects.mt.product import TranslateWebSnapshot
from sandbox.projects.mt.util.arc_mixin import ArcMixin
from sandbox.projects.mt.util.mt_web import MtWebMixin
from sandbox.projects.mt.util.task_profiler import TaskProfilerMixin
from sandbox.sdk2 import yav


class ReleaseTranslateWeb(TaskProfilerMixin, ArcMixin, MtWebMixin, nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Build MT Web package for release to beta
    """

    class Requirements(sdk2.Requirements):
        cores = 4
        ram = 8000
        disk_space = 16000
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Release params') as release_params_group:
            arc_params = ArcMixin.ArcParams()

        with sdk2.parameters.Group('Caching params', collapse=True) as caching_params_group:
            caching_params = MtWebMixin.CachingParams()

        with sdk2.parameters.Group('Advanced task params', collapse=True) as advanced_task_params_group:
            process_params = MtWebMixin.ProcessParams()

        with sdk2.parameters.Group('BuildParams'):
            build_params = MtWebMixin.BuildParams()

        with sdk2.parameters.Output():
            static_version = sdk2.parameters.String('Static version')

        _container = sdk2.parameters.Container(
            label='Translate container',
            resource_type=TranslateWebLxcContainer,
            platform='linux_ubuntu_16.04_xenial',
            required=True,
        )


    class Context(TaskProfilerMixin.Context, MtWebMixin.Context, sdk2.Context):
        pass

    @sdk2.footer()
    def footer(self):
        return self.get_profiler_html(task=self)

    def on_execute(self):
        super(ReleaseTranslateWeb, self).on_execute()
        self.init_mt_web(task=self)
        self.init_arc(task=self)
        self.init_task_profiler(task=self)

        try:
            with self.profiler.action('Mount Arc'):
                self.mount_arc()

            self.run_main_actions()
        finally:
            with self.profiler.action('Unmount Arc'):
                self.unmount_arc()

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)

    def run_main_actions(self):
        with self.profiler.action('Setup venv'):
            self.setup_venv()

        with self.profiler.action('Collect node_modules from cache'):
            self.collect_node_modules_from_cache()

        with self.profiler.action('Build'):
            version = self.build_project()

        with self.profiler.action('Upload static'):
            self.upload_static(version)

        with self.profiler.action('Publish package'):
            self.publish_package(version)

        with self.profiler.action('Cache node_modules'):
            self.store_node_modules_to_cache()

        self.Parameters.static_version = version

    def build_project(self):
        version_type, release_version = self.get_version_rule()
        s3_host = self.get_s3_host()
        out = self.run_project_method(
            'ci-build-for-release',
            '--version-type=' + version_type,
            '--release-version=' + str(release_version),
            '--static-host=' + s3_host
        )
        return self.get_var_from_method_out(out, 'static_version')

    def upload_static(self, version):
        access_key, secret_key = self.get_s3_credentials()

        s3_secret_dir = os.path.join(self.work_dir, '.s3')
        access_key_file = os.path.join(s3_secret_dir, 'access_key')
        secret_key_file = os.path.join(s3_secret_dir, 'secret_key')
        s3_host = self.get_s3_host()

        if not os.path.exists(s3_secret_dir):
            os.makedirs(s3_secret_dir)

        with open(access_key_file, 'w') as a_fp, open(secret_key_file, 'w') as s_fp:
            a_fp.write(access_key)
            s_fp.write(secret_key)

        self.run_project_method('ci-upload-static', '--static-version=' + version, '--static-host=' + s3_host)

    def get_s3_credentials(self):
        try:
            # noinspection SpellCheckingInspection
            secret = yav.Secret('sec-01dfe0sqjz34jk83hhkhh6vzd0')
            data = secret.data()

            access_key = data['AccessKey']
            secret_key = data['SecretKey']
        except BaseException as e:
            logging.error(e)
            access_key = sdk2.Vault.data('MT', 'S3_ACCESS_KEY')
            secret_key = sdk2.Vault.data('MT', 'S3_SECRET_KEY')

        return access_key, secret_key

    def get_s3_host(self):
        return 'prod' if self.Parameters.use_s3_production_bucket else 'beta'

    def get_version_rule(self):
        if self.Parameters.use_external_version:
            return 'manual', self.Parameters.release_version
        else:
            return 'commit', '0'

    def publish_package(self, version):
        dest_dir = os.path.join(self.work_dir, 'mt')
        out = self.run_project_method('ci-get-ignore-patterns')

        ignore_patterns = self.make_list_from_method_out(out, replace_dir=False)
        ignore_patterns.append('v' + version)
        logging.debug('Package ignore patterns: %s', ignore_patterns)

        ignore = shutil.ignore_patterns(*ignore_patterns)
        shutil.copytree(self.project_dir, dest_dir, ignore=ignore)

        res = TranslateWebSnapshot(
            self, 'Release web {}'.format(version), dest_dir,
            commit=self.Parameters.commit,
        )
        sdk2.ResourceData(res).ready()
