# coding=utf-8

from __future__ import absolute_import

import sandbox.common.types.client as ctc
import datetime
from sandbox import sdk2


class SpellcheckerRemDailyBinary(sdk2.Resource):
    """ `rem_daily` binary """
    executable = True


class SpellcheckerRemDailyConfig(sdk2.Resource):
    """ `rem_daily` config """
    pass


class SpellcheckerRemDailyParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group("rem_daily parameters"):
        rem_daily_executable_r = sdk2.parameters.Resource(
            'binary',
            resource_type=SpellcheckerRemDailyBinary,
            required=True, )
        rem_daily_config_r = sdk2.parameters.Resource(
            'config',
            resource_type=SpellcheckerRemDailyConfig,
            required=True, )
        rem_daily_ignore_duplicate_package_error = sdk2.parameters.Bool(
            'ignore duplicate package error',
            default_value=False, )
        rem_daily_date = sdk2.parameters.String(
            'date in YYYY-MM-DD format (will use current date if not specified)',
            default_value='', )
        rem_daily_token = sdk2.parameters.String(
            'name of the secret in the Vault with REM OAuth token',
            required=True, )


def _validate_date_string(s):
    try:
        datetime.datetime.strptime(s, '%Y-%m-%d')
    except ValueError:
        raise ValueError('Incorrect data format, expected YYYY-MM-DD')


class RunSpellcheckerRemDaily(sdk2.Task):
    """ Plan daily spellchecker data pipeline """

    class Requirements(sdk2.Requirements):
        client_tags = (ctc.Tag.GENERIC | ctc.Tag.LXC) & ctc.Tag.Group.LINUX

    class Parameters(SpellcheckerRemDailyParameters):
        pass

    def on_enqueue(self):
        if self.Parameters.rem_daily_date:
            # can we move parameter validation to the earlier stage?
            _validate_date_string(self.Parameters.rem_daily_date)

    def on_execute(self):
        date = self.Parameters.rem_daily_date
        if not date:
            date = datetime.date.today().strftime('%Y-%m-%d')

        rem_daily_path = str(sdk2.ResourceData(self.Parameters.rem_daily_executable_r).path)
        rem_daily_cfg = str(sdk2.ResourceData(self.Parameters.rem_daily_config_r).path)
        rem_token = sdk2.Vault.data(self.Parameters.rem_daily_token)

        with sdk2.helpers.ProcessLog(self, logger='rem_daily') as log:
            sdk2.helpers.subprocess.check_call(
                (rem_daily_path, 'show_version', ), stdout=log.stdout, stderr=log.stderr, )

        cmd = [rem_daily_path, 'add', '--config', rem_daily_cfg, '--date', date, ]
        if self.Parameters.rem_daily_ignore_duplicate_package_error:
            cmd.append('--ignore-duplicate-package-error')

        with sdk2.helpers.ProcessLog(self, logger='rem_daily') as log:
            sdk2.helpers.subprocess.check_call(
                cmd, stdout=log.stdout, stderr=log.stderr,
                env={'REM_TOKEN': rem_token, })
