# coding=utf-8

import os
import json
import logging

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.mt.product import TranslateWebLxcContainer
from sandbox.projects.mt.util.arc_mixin import ArcMixin
from sandbox.projects.mt.util.mt_web import MtWebMixin
from sandbox.projects.mt.util.task_profiler import TaskProfilerMixin

import requests


class TranslateAvailabilityTask(TaskProfilerMixin, ArcMixin, MtWebMixin, sdk2.Task):
    """
    Check service availability task
    """

    class Requirements(sdk2.Requirements):
        cores = 4
        ram = 8000
        disk_space = 16000
        dns = ctm.DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Output():
            errors = sdk2.parameters.String('errors')

        with sdk2.parameters.Group('Arc params') as arc_group:
            arc_params = ArcMixin.ArcParams()

        with sdk2.parameters.Group('Availability params', collapse=True) as availability_group:
            urls = sdk2.parameters.List('Additional urls')
            check_manifest = sdk2.parameters.Bool('Check urls from yandex-manifest.json', default_value=True)
            manifest_url = sdk2.parameters.String(
                'yandex-manifest.json absolute url',
                default_value='https://translate.yandex.ru/yandex-manifest.json',
            )
            host = sdk2.parameters.String(
                'Host target',
                ui=sdk2.parameters.String.UI('select'),
                choices=[('production', 'production'), ('beta', 'beta')],
                default_value='beta'
            )

        with sdk2.parameters.Group('Caching params', collapse=True) as caching_params_group:
            caching_params = MtWebMixin.CachingParams()

        with sdk2.parameters.Group('Advanced task params', collapse=True) as advanced_task_params_group:
            process_params = MtWebMixin.ProcessParams()

        _container = sdk2.parameters.Container(
            label='Translate container',
            resource_type=TranslateWebLxcContainer,
            platform='linux_ubuntu_16.04_xenial',
            required=True,
        )

    def on_execute(self):
        super(TranslateAvailabilityTask, self).on_execute()
        self.init_mt_web(task=self)
        self.init_arc(task=self)
        self.init_task_profiler(task=self)
        self.error_urls = []

        try:
            with self.profiler.action('Mount Arc'):
                self.mount_arc()

            self.run_main_actions()
        finally:
            with self.profiler.action('Unmount Arc'):
                self.unmount_arc()

    def run_main_actions(self):
        with self.profiler.action('Setup venv'):
            self.setup_venv()

        with self.profiler.action('Get changelog & create release ticket'):
            self.check_service_availability()

    def check_service_availability(self):
        self.test_manifest_urls()
        self.test_urls()
        if len(self.error_urls) > 0:
            self.Parameters.errors = '\n'.join(list(map(lambda u: '× ' + u, self.error_urls)))
            raise TaskFailure('Some urls are not available!')

    def test_manifest_urls(self):
        manifest = self.Parameters.manifest_url
        logging.info('test urls from {}'.format(manifest))
        response = requests.get(manifest, verify=False)
        data = response.json()
        urls = data['yandex']['cache']['resources']
        self._check_urls(
            urls=map(lambda u: 'https:' + u if u.startswith("//") else u, urls)
        )

    def test_urls(self):
        config_file = os.path.join(self.project_dir, 'tools', 'release', 'availability.json')
        logging.info('test urls from config {}'.format(config_file))
        with open(config_file) as f:
            config = json.load(f)
        config_urls = config[self.Parameters.host]
        urls = config_urls + list(self.Parameters.urls)
        self._check_urls(urls)

    def _check_urls(self, urls):
        for url in urls:
            logging.debug('get {}'.format(url))
            response = requests.get(url, verify=False)
            if response.status_code != 200:
                self.error_urls.append(url)
                logging.info('× {}'.format(url))
            else:
                logging.info('✓ {}'.format(url))
