# encoding=utf-8

from __future__ import absolute_import

import distutils
import distutils.dir_util
import distutils.file_util
import logging
import os
from sandbox import sdk2
import sandbox.common.types.resource as ctr

_logger = logging.getLogger(__name__)


def _copy_file(src, dst, logger):
    try:
        distutils.file_util.copy_file(src, dst, link='hard')
    except OSError as e:
        if e.errno == 18:
            logger.debug('"%s" is on the different device, will copy instead', dst)
            distutils.file_util.copy_file(src, dst)
        else:
            raise


def copy_tree_using_hardlinks(src, dst, override=False, logger=None):
    """
    Copy an entire directory `src` to a new location `dst` using hardlinks. Both `src` and `dst`
    must be directory names.

    If `override` is `True` will override files in `dst`, otherwise will skip collisions.

    This function is usefull when you are making a copy of a large resource and only need to modify
    a couple files in it. Or when you assemble resource from a number of its parts.

    :param src: Source directory.
    :param dst: Destination directory.
    :param override: Override files in destination directory.
    :type override: bool
    """
    if logger is None:
        logger = _logger

    logger.debug('about to copy all content of %s into %s', src, dst)

    for root, dirs, files in os.walk(src):
        for name in files:
            src_path = os.path.join(root, name)
            rel_path = os.path.relpath(src_path, src)
            logger.debug('looking at "%s"', rel_path)

            dst_path = os.path.join(dst, rel_path)
            if os.path.isfile(dst_path):
                if override:
                    logger.debug('removing "%s"', rel_path)
                    os.path.remove(dst_path)
                else:
                    logger.debug('skipping "%s"', rel_path)
                    continue

            distutils.dir_util.mkpath(os.path.dirname(dst_path))

            logger.debug('hardlinking "%s"', rel_path)
            _copy_file(src_path, dst_path, logger=logger)


def override_file_with_resource(dst_rel_path, dst_dir,
                                resource_type=None,
                                resource_attributes=None,
                                resource_id=None,
                                logger=None):
    """
    Override `dst_rel_path` with file from resource. If `resource_id` wasn't provided will find last
    resource of type `resource_type` with attributes `resource_attributes`.

    :param resource_type: Type of sandbox resource e.g. `SpellcheckerSnapshotPart`
    :param resource_attributes: Attrites of resource.
    :type resource_attributes: dict
    :param resource_id: Resource id.
    :type resource_id: int
    :param dst_rel_path: Relative path of the file within destination directory.
    :type dst_rel_path: str
    :param dst_dir: Destination directory.
    :type dst_dir: str
    """
    if logger is None:
        logger = _logger

    if resource_id is None:
        assert resource_type is not None
        assert resource_attributes is not None

        resource_id = resource_type.find(
            state=ctr.State.READY,
            attrs=resource_attributes, ).first()
    else:
        assert resource_type is not None
        assert resource_attributes is not None

    src_path = sdk2.ResourceData(resource_id).path

    dst_path = os.path.join(dst_dir, dst_rel_path)
    dst_dir = os.path.dirname(dst_path)

    distutils.dir_util.mkpath(dst_dir)

    if os.path.isfile(dst_path):
        logger.debug('removing old "%s"', dst_rel_path)
        os.remove(dst_path)

    logger.debug('hardlinking "%s"', dst_rel_path)
    _copy_file(str(src_path), dst_path, logger=logger)


def try_get_resource_html_map(task, resource):
    """
    Build HTML map of resource.

    :param task: Task that invokes this function
    :type task: sdk.Task
    :param resource: Resource to build map for.
    :type resource: sdk2.Resource
    """
    cmd = (
        'tree',
        '--du',
        '--charset', 'utf-8',
        '--dirsfirst',
        '-H', resource.http_proxy,
        str(resource.path),),

    timeout_sec = 3 * 60  # 3 minutes

    from sdk2.helpers import subprocess

    html = None
    success = False
    try:
        with sdk2.helpers.ProcessLog(task, logger=_logger) as log:
            html = subprocess.checkout(cmd, stderr=log.stderr, timeout=timeout_sec)
        success = True
    except subprocess.SubprocessError:
        _logger.exception('ignored')

    return success, html


def try_make_resource_html_map_resource(task, resource):
    """
    TODO(yazevnul): implement
    """
    pass
