import hashlib
import logging
import os
import re
import shutil
import subprocess
import tarfile
import time
from multiprocessing.pool import ThreadPool

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.resource import State
from sandbox.projects.mt.product import MtWebNodeModules, MtWebPythonReqs, MtWebTestArtifact


class MtWebMixin(object):
    class CachingParams(sdk2.Parameters):
        read_venv_cache = sdk2.parameters.Bool('Read venv cache', default_value=True)
        read_npm_cache = sdk2.parameters.Bool('Read npm cache', default_value=True)
        write_venv_cache = sdk2.parameters.Bool('Write venv cache', default_value=True)
        write_npm_cache = sdk2.parameters.Bool('Write npm cache', default_value=True)

    class ServiceParams(sdk2.Parameters):
        launch_service = sdk2.parameters.Bool('Launch service', required=True, default_value=False)
        launch_wait = sdk2.parameters.Float('Launch wait seconds', required=True, default_value=0.0)

    class ProcessParams(sdk2.Parameters):
        env = sdk2.parameters.Dict('Env vars')

    class BuildParams(sdk2.Parameters):
        use_s3_production_bucket = sdk2.parameters.Bool(
            'Publish static sources to production bucket',
            required=True,
            default_value=False
        )
        use_external_version = sdk2.parameters.Bool(
            'Use version from release_version arg (otherwise it will be generated)',
            required=True,
            default_value=False
        )
        release_version = sdk2.parameters.String(
            'Release version (from release branch)',
            required=False,
            default_value=None
        )

    class Context(sdk2.Context):
        service_logs = ''

    def init_mt_web(self, task):
        self.work_dir = os.getcwd()
        self.__task = task

        self.repo_path = os.path.join(self.work_dir, 'arc')
        self.project_dir = os.path.join(self.repo_path, str(self.__task.Parameters.arcadia_path))
        self.venv_dir = os.path.join(self.work_dir, 'venv')
        self.log_dir = os.path.join(self.project_dir, 'dev_data', 'logs')

        self.package_files = []
        self.node_module_caches_used = set()
        self.log_file_descriptors = []

    def finalize_mt_web(self):
        self.close_log_files()

    def setup_venv(self):
        requirements_hash = self.get_requirements_hash()
        has_cache = self.collect_venv_from_cache(requirements_hash)
        if not has_cache:
            self.run_project_method('setup-venv', '--rebuild-venv')
            self.store_venv_to_cache(requirements_hash)

    def get_requirements_hash(self):
        out = self.run_project_method('ci-get-requirement-files')
        req_files = self.make_list_from_method_out(out)

        text_parts = []
        for file_path in req_files:
            req_file = os.path.join(self.project_dir, file_path)
            if not os.path.isfile(req_file):
                raise TaskFailure('There is no ' + req_file)

            with open(req_file) as fp:
                text_parts.append(fp.read())

        text = ''.join(text_parts)
        return hashlib.sha256(text).hexdigest()

    def collect_venv_from_cache(self, requirements_hash):
        if not self.__task.Parameters.read_venv_cache:
            return False

        res = sdk2.Resource.find(
            type=MtWebPythonReqs,
            state=State.READY,
            attrs={'hash': requirements_hash},
        ).order(-sdk2.Resource.id).first()

        if not res:
            return False

        with tarfile.open(str(sdk2.ResourceData(res).path), 'r:gz') as tar:
            tar.extractall(self.venv_dir)

        reuse_msg = 'Reuse venv resource <a href="/resource/{0}/view" target="_blank">#{0}</a>'.format(res.id)
        self.__task.set_info(reuse_msg, do_escape=False)

        return True

    def store_venv_to_cache(self, requirements_hash):
        if not self.__task.Parameters.write_venv_cache:
            return

        archive_name = 'venv.tar.gz'
        with tarfile.open(archive_name, 'w:gz') as tar:
            tar.add(self.venv_dir, '.')

        res = MtWebPythonReqs(self, 'MT Web Python venv', archive_name, hash=requirements_hash)
        sdk2.ResourceData(res).ready()

    def collect_package_files(self):
        out = self.run_project_method('ci-get-package-files')
        self.package_files = self.make_list_from_method_out(out)

    def collect_node_modules_from_cache(self):
        self.collect_package_files()

        if not self.__task.Parameters.read_npm_cache:
            return

        messages = []
        for reuse_msg in ThreadPool().map(self.unpack_cached_node_modules, self.package_files):
            if reuse_msg:
                messages.append(reuse_msg)

        if messages:
            messages.sort()
            self.__task.set_info('<br>'.join(messages), do_escape=False)

    def unpack_cached_node_modules(self, file_path):
        res = sdk2.Resource.find(
            type=MtWebNodeModules,
            state=State.READY,
            attrs={'hash': self.get_node_modules_hash(file_path)},
        ).order(-sdk2.Resource.id).first()

        if not res:
            return None

        arch_path = str(sdk2.ResourceData(res).path)
        node_modules = os.path.join(os.path.dirname(file_path), 'node_modules')

        if not os.path.exists(node_modules):
            os.makedirs(node_modules)

        # Using external tar for parallel unpack
        self.check_output(['tar', '-xzf', arch_path, '-C', node_modules])
        self.node_module_caches_used.add(node_modules)

        return 'Reuse {0} resource <a href="/resource/{1}/view" target="_blank">#{1}</a>'.format(
            node_modules.replace(self.project_dir + '/', ''),
            res.id,
        )

    def store_node_modules_to_cache(self):
        if not self.package_files:
            self.collect_package_files()

        if not self.__task.Parameters.write_npm_cache:
            return

        to_pack = []
        for file_path in self.package_files:
            node_modules = os.path.join(os.path.dirname(file_path), 'node_modules')
            if not os.path.isdir(node_modules) or node_modules in self.node_module_caches_used:
                continue

            to_pack.append(file_path)

        if to_pack:
            ThreadPool().map(self.pack_node_modules_to_cache, to_pack)

    def pack_node_modules_to_cache(self, file_path):
        logging.info('Cache node_modules for %s', file_path)

        arch_name = 'node_modules_%s.tar.gz' % hashlib.md5(file_path).hexdigest()
        arch_path = os.path.join(self.work_dir, arch_name)
        node_modules = os.path.join(os.path.dirname(file_path), 'node_modules')

        # Using external tar for parallel pack
        self.check_output(['tar', '-czf', arch_name, '-C', node_modules, '.'])

        cache_key = self.get_node_modules_hash(file_path)
        node_modules_relative = node_modules.replace(self.project_dir + '/', '')

        res = MtWebNodeModules(self, node_modules_relative, arch_path, hash=cache_key)
        sdk2.ResourceData(res).ready()

    def get_node_modules_hash(self, package_path):
        relative_path = package_path.replace(self.project_dir + '/', '')

        with open(package_path) as fp:
            package_content = fp.read()

        npm_root = os.path.dirname(package_path)
        lock_path = os.path.join(npm_root, 'package-lock.json')

        if os.path.isfile(lock_path):
            with open(lock_path) as fp:
                lock_content = fp.read()
        else:
            lock_content = ''
            logging.warning('There is no package-lock.json for %s', package_path)

        text = relative_path + ':' + package_content + ':' + lock_content
        return hashlib.sha256(text).hexdigest()

    def launch_service_if_needed(self):
        service_p = None
        if self.__task.Parameters.launch_service:
            service_cmd = self.get_entrypoint()
            service_p = self.run_process(service_cmd, tag='service')
            time.sleep(float(self.__task.Parameters.launch_wait))

        return service_p

    def stop_service(self, service_p):
        if service_p is None:
            return

        service_p.terminate()
        if service_p.returncode:
            logging.warning('Service return code is %s', service_p.returncode)

    def get_entrypoint(self):
        out = self.run_project_method('ci-get-entrypoint')
        result = self.make_list_from_method_out(out)
        if not result:
            raise TaskFailure('Action "ci-get-entrypoint" returned an empty entrypoint')

        return result

    def run_process(self, command, tag=None):
        logging.info('Launch [%s]: %s', tag, command)

        if tag is None:
            stdout = subprocess.PIPE
            stderr = subprocess.PIPE
        else:
            stdout = self.register_log_file(os.path.join(self.log_dir, '%s.out.log' % tag))
            stderr = self.register_log_file(os.path.join(self.log_dir, '%s.err.log' % tag))

        return subprocess.Popen(
            command,
            stdout=stdout,
            stderr=stderr,
            env=self.get_subprocess_environment(),
        )

    def check_output(self, command):
        p = self.run_process(command)
        out, err = p.communicate()
        ret_code = p.returncode

        logging.debug('Command "%s" results:\nSTDERR: %s\nSTDOUT: %s', command, err, out)
        if ret_code:
            raise TaskFailure('Command "%s" failed with status %s. Check debug.log' % (command, ret_code))

        return out

    def run_project_method(self, *run_args):
        run_script = os.path.join(self.project_dir, 'run')
        if not os.path.isfile(run_script):
            raise TaskFailure('There is no "run" script in project root')

        return self.check_output([run_script] + list(run_args))

    def make_list_from_method_out(self, out, replace_dir=True):
        result = []
        for part in out.split('\n'):
            val = part.strip()
            if not val:
                continue

            if replace_dir:
                val = re.sub(r'^\.', self.project_dir, val)

            result.append(val)

        return result

    def get_var_from_method_out(self, out, var_name, replace_dir=True, absent_ok=False):
        lines = self.make_list_from_method_out(out, replace_dir=replace_dir)

        var_pattern = re.compile('### %s=(.+?) ###' % re.escape(var_name))
        for line in lines:
            match = var_pattern.match(line)
            if match:
                return match.group(1)

        if absent_ok:
            return None

        raise ValueError('There is no variable %s in command output' % var_name)

    def get_subprocess_environment(self):
        process_env = os.environ.copy()
        process_env.update({
            'PATH': os.path.join(self.venv_dir, 'bin') + ':' + os.getenv('PATH', ''),
            'HOME': self.work_dir,
            'VENV_DIR': self.venv_dir,
            'PROJECT_DIR': self.project_dir,
            'MT_NPM_USE_CI': '1',
            'CI_ENV': '1',
        })
        process_env.update(self.__task.Parameters.env)

        logging.debug('Process environment: %s', process_env)
        return process_env

    def register_log_file(self, file_name):
        fp = open(file_name, 'w')
        self.log_file_descriptors.append(fp)
        return fp

    def close_log_files(self):
        for fp in self.log_file_descriptors:
            fp.close()
        self.log_file_descriptors = []

    def save_logs(self, commit, flow_launch_id='None'):
        export_log_dir = os.path.join(self.work_dir, 'project_logs')
        shutil.copytree(self.log_dir, export_log_dir, ignore=shutil.ignore_patterns('.arcignore'))

        res = MtWebTestArtifact(
            self, 'Test logs', export_log_dir,
            commit=commit, flow_launch_id=flow_launch_id, type='logs',
        )
        sdk2.ResourceData(res).ready()

        self.Context.service_logs = str(res.http_proxy)
