import logging
import time
from collections import namedtuple
from datetime import datetime

# noinspection PyUnresolvedReferences
from aniso8601 import parse_datetime
from sandbox.common.types.misc import NotExists
from sandbox import sdk2

class TaskProfiler(object):
    def __init__(self, data):
        self._data = data

    def action(self, name):
        return TaskProfilerAction(name, self._data)

class TaskProfilerAction(object):
    def __init__(self, name, data):
        self._name = name
        self._data = data
        self._start_time = None

    def __enter__(self):
        self._start_time = time.time()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self._data.append(ActionTimeRecord(self._name, self._start_time, time.time()))


ActionTimeRecord = namedtuple('ActionTimeRecord', ('name', 'start_time', 'end_time'))


class TaskProfilerMixin(object):
    class Context(sdk2.Context):
        profiler_data = []

    def get_profiler_html(self, task):
        self.__task = task
        try:
            profile = self._get_combined_profile()
            return self._render_profile_html(profile)
        except BaseException as e:
            logging.error(e)
            return 'Error when trying to get profile data: ' + str(e)

    def init_task_profiler(self, task):
        self.__task = task

        if self.Context.profiler_data is NotExists:
            self.Context.profiler_data = []

        self.__profiler = TaskProfiler(self.Context.profiler_data)

    @property
    def profiler(self):
        try:
            return self.__profiler
        except AttributeError:
            raise RuntimeError('Profiler is not inited. You should call self.init_task_profiler(task=self)')

    def _get_combined_profile(self):
        profile = self._get_system_profile()
        profile.extend(self._get_custom_profile())

        profile.sort(key=lambda x: str(x.start_time))
        return profile

    def _get_system_profile(self):
        status_data = []
        audit_data = self.__task.server.task[self.__task.id].audit.read()
        for item in audit_data:
            status = item.get('status')
            start_time = item.get('time')

            if not status or not start_time:
                continue

            start_dt = parse_datetime(start_time)
            status_data.append((status, start_dt))

        system_profile = []
        status_count = len(status_data)
        for i in range(1, status_count):
            status, start_time = status_data[i - 1]
            _, end_time = status_data[i]

            system_profile.append(ActionTimeRecord(status, start_time, end_time))

        if status_count > 2:
            status, start_time = status_data[-1]
            system_profile.append(ActionTimeRecord(status, start_time, start_time))

        return system_profile

    def _get_custom_profile(self):
        if self.Context.profiler_data is NotExists:
            return []

        profile = []
        for status, start_time, end_time in self.Context.profiler_data:
            profile.append(ActionTimeRecord(
                status,
                datetime.utcfromtimestamp(start_time),
                datetime.utcfromtimestamp(end_time),
            ))

        return profile

    def _render_profile_html(self, profile):
        parts = [
            '<strong style="font-size: 1.2em; margin: 0 0 0.5em 0">Profiler result</strong>',
            '<table style="margin: 0">',
        ]

        for name, start_time, end_time in profile:
            parts.append(
                '<tr><td style="padding-right: 1em">{}</td><td>{}</td></tr>'.format(
                    name, end_time - start_time,
                )
            )

        parts.append('</table>')
        return ''.join(parts)
