# coding: utf-8

import os
import shutil
import logging
import tarfile
import tempfile
from sandbox import sdk2

from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6
import sandbox.projects.common.constants as consts
import sandbox.common.types.task as ctt
from sandbox import common


class YamrecUpperResource(sdk2.Resource):
    """
    .tar.gz file containing yamrec upper application code
    """
    releasers = ['robot-muzsearch', 'nglaz', 'singleton', 'denshv']
    release_subscribers = []
    releasable = True


class YamrecUpperTensorflowModel(sdk2.Resource):
    """
    .tar.gz file containing model and model.meta files with Tensorflow model for yet another music recognition (yamrec)
    """
    releasable = False
    share = True


class YamrecUpperDockerfileDir(sdk2.Resource):
    """
    Directory containing dockerfile to build docker image with environment for yamrec upper application
    """
    releasable = False
    share = False


class BuildYamrecUpper(sdk2.Task):
    """
    Builds yamrec upper search application code as .tar.gz and its environment as docker image
    """

    REGISTRY_TAG = 'music/yamrec/upper-env'

    SRC_DIR = '/extsearch/audio/yamrec/upper'
    ADDITIONAL_SRCS = (
        '/extsearch/audio/yamrec/scripts/make_mel.py',
        '/extsearch/audio/yamrec/scripts/search_result.py',
        '/yweb/music/pylib/asynclog.py',
        '/yweb/music/pylib/yconf.py'
    )

    class Parameters(sdk2.Task.Parameters):
        svn_url_for_arcadia = sdk2.parameters.String('Svn url for arcadia, HEAD by default', default="arcadia:/arc/trunk/arcadia")
        build_docker_image = sdk2.parameters.Bool("Build environment docker image", default=True)
        with build_docker_image.value[True]:
            docker_file_url = sdk2.parameters.String('Dockerfile url', required=True)

    class Context(sdk2.Context):
        child_tasks_ids = []
        yamrec_upper_resource_id = None
        docker_task_id = None

    def on_prepare(self):
        self.Context.checkout_path = os.path.join(self.path(), tempfile.mkdtemp())
        self._checkout_files()

    def on_execute(self):
        child_tasks_ids = self.Context.child_tasks_ids
        if not child_tasks_ids:
            self._create_subtasks()
        resource_id = self.Context.yamrec_upper_resource_id
        if not resource_id:
            subtasks = self.server.task.read(id=child_tasks_ids, children=True, limit=len(child_tasks_ids))["items"]
            logging.info("Subtasks: {}".format(subtasks))
            failed_subtasks = [subtask for subtask in subtasks if subtask["status"] not in ctt.Status.Group.SUCCEED]
            if failed_subtasks:
                raise common.errors.TaskFailure("Failed subtasks: {}".format(failed_subtasks))

            work_path = os.path.join(self.path(), tempfile.mkdtemp())
            self._copy_svn_files(work_path)
            self._copy_subtask_resources(work_path)
            resource = YamrecUpperResource(self, '.tar.gz containing scripts for upper yamrec search', 'yamrec_upper.tar.gz')
            resource_data = sdk2.ResourceData(resource)
            with tarfile.open(str(resource_data.path), 'w:gz') as tar:
                for root, _, files in os.walk(work_path):
                    for filename in files:
                        tar.add(os.path.join(root, filename), arcname=filename)
            self.Context.yamrec_upper_resource_id = str(resource.id)
            resource_data.ready()
        docker_task_id = self.Context.docker_task_id
        if self.Parameters.build_docker_image and not docker_task_id:
            dockerfile_resource = YamrecUpperDockerfileDir(self, 'directory containing dockerfile for {} image'.format(self.REGISTRY_TAG), 'docker')
            dockerfile_resource_data = sdk2.ResourceData(dockerfile_resource)
            dockerfile_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
            self._export_dockerfile(str(dockerfile_resource_data.path))
            dockerfile_resource_data.ready()
            self._create_docker_image(dockerfile_resource.id)
        elif docker_task_id:
            subtask = self.server.task.read(id=docker_task_id, children=True, limit=1)["items"][0]
            logging.info("Docker build subtask: {}".format(subtask))
            if subtask["status"] not in ctt.Status.Group.SUCCEED:
                raise common.errors.TaskFailure("Failed subtask: {}".format(subtask))

    def _checkout_files(self):
        parsed_url = sdk2.svn.Arcadia.parse_url(self.Parameters.svn_url_for_arcadia)
        directory_url = sdk2.svn.Arcadia.replace(self.Parameters.svn_url_for_arcadia, path=parsed_url.path + self.SRC_DIR)
        sdk2.svn.Arcadia.checkout(directory_url, self.Context.checkout_path)
        for f in self.ADDITIONAL_SRCS:
            file_url = sdk2.svn.Arcadia.replace(self.Parameters.svn_url_for_arcadia, path=parsed_url.path + f)
            sdk2.svn.Arcadia.export(file_url, self.Context.checkout_path)

    def _create_subtask(self, params, task_type, description):
        task = self.server.task({
            'type': task_type,
            'context': params,
            'children': True,
        })
        self.server.task[task['id']].update({
            'description': description,
            'owner': self.owner,
            'notifications': self.server.task[self.id].read()["notifications"],
            'priority': {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.scls,
            },
        })
        self.server.batch.tasks.start.update([task['id']])
        return task['id']

    def _create_subtasks(self):
        httpsearchclient_params = {
            'checkout_arcadia_from_url': self.Parameters.svn_url_for_arcadia,
            'targets': 'bindings/python/httpsearchclient',
            'arts': 'bindings/python/httpsearchclient/httpsearchclient.so',
            'build_system': 'ya',
            'build_type': 'release',
            'check_return_code': True,
            'result_single_file': True,
            consts.DEFINITION_FLAGS_KEY: '-DWERROR_MODE=none -DUSE_SYSTEM_PYTHON=2.7',
        }
        meta_pb2_params = {
            'checkout_arcadia_from_url': self.Parameters.svn_url_for_arcadia,
            'targets': 'search/idl/python',
            'arts': 'search/idl/meta_pb2.py',
            'build_system': 'ya',
            'build_type': 'release',
            'check_return_code': True,
            'result_single_file': True,
        }
        self.Context.child_tasks_ids = [
            self._create_subtask(httpsearchclient_params, 'YA_MAKE', 'Build httpsearchclient.so for task {}'.format(self.id)),
            self._create_subtask(meta_pb2_params, 'YA_MAKE', 'Build meta_pb2.py for task {}'.format(self.id)),
        ]
        waited_statuses = set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
        raise sdk2.WaitTask(self.Context.child_tasks_ids, waited_statuses, wait_all=True)

    def _create_docker_image(self, resource_id):
        build_docker_image_v6_params = {
            BuildDockerImageV6.PackagedResource.name: resource_id,
            BuildDockerImageV6.RegistryTags.name: [self.REGISTRY_TAG],
            BuildDockerImageV6.RegistryLogin.name: self.author,
            BuildDockerImageV6.VaultItemName.name: 'docker_token',
            BuildDockerImageV6.VaultItemOwner.name: 'robot-muzsearch',
        }
        self.Context.docker_task_id = self._create_subtask(build_docker_image_v6_params, 'BUILD_DOCKER_IMAGE_V6', 'Build docker image for task {}'.format(self.id))
        waited_statuses = set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
        raise sdk2.WaitTask(self.Context.docker_task_id, waited_statuses, wait_all=True)

    def _copy_svn_files(self, dst_path):
        for root, _, files in os.walk(self.Context.checkout_path):
            for f in files:
                if f.endswith('.py') and (not f.startswith('test')) and f != 'ya.make':
                    shutil.copy(os.path.join(root, f), dst_path)

    def _copy_subtask_resources(self, dst_path):
        for child_task_id in self.Context.child_tasks_ids:
            resource = list(sdk2.Resource.find(task_id=child_task_id, type='ARCADIA_PROJECT').limit(1))[0]
            resource_data = sdk2.ResourceData(resource)
            logging.info('Copying resource from ' + str(resource_data.path))
            shutil.copy(str(resource_data.path), dst_path)

    def _export_dockerfile(self, dst_path):
        sdk2.svn.Arcadia.export(self.Parameters.docker_file_url, dst_path)
        os.rename(
            os.path.join(dst_path, self.Parameters.docker_file_url[self.Parameters.docker_file_url.rfind('/')+1:]),
            os.path.join(dst_path, 'Dockerfile')
        )
