import hashlib
import logging
import multiprocessing
import os
import platform
import shutil
import sys
from glob import glob

import requests


class DummyProcessLog:
    def __init__(self, dummy, logger):
        self.stdout = sys.stdout

    def __enter__(self, *args):
        return self

    def __exit__(self, *args):
        return


try:
    from sandbox.sdk2.helpers import subprocess as sp
    from sandbox.sdk2.helpers import ProcessLog
except ImportError:
    import subprocess as sp

    ProcessLog = DummyProcessLog

DOWNLOAD_DIR = 'download'
BUILD_DIR = 'build'
TARGET_DIR = 'build-target'

CPU_COUNT = str(multiprocessing.cpu_count())


class Buildable(object):
    def __init__(self, name, url, md5, file='', configure_args=None):
        self.log = logging.getLogger(self.__class__.__name__)
        self.name = name
        self.url = url
        self.md5 = md5
        self.file = file if file else url.split('/')[-1]
        self.configure_args = configure_args if configure_args is not None else ['--enable-static', '--disable-shared']

        self._sed = 'gsed' if platform.system() == 'Darwin' else 'sed'

        self.target_dir = os.path.abspath(TARGET_DIR)
        self.task_id = 0

        self._build_cache = None

    @property
    def target_file(self):
        return '%s/%s' % (DOWNLOAD_DIR, self.file)

    @property
    def build_dir(self):
        if self._build_cache is not None:
            return self._build_cache

        n = glob('%s/%s*' % (BUILD_DIR, self.name))
        if len(n) > 1:
            raise RuntimeError('Got multiple dirs globbing ' + self.name)

        if n:
            self._build_cache = n[0]
            return n[0]
        return None

    def download(self):
        target_file = self.target_file
        if os.path.exists(target_file):
            return

        self.log.info('Downloading %s', target_file)
        r = requests.get(self.url, stream=True)
        r.raise_for_status()
        with open(target_file, 'wb') as f:
            for chunk in r.iter_content(chunk_size=1024):
                f.write(chunk)

        if self.md5 is None:
            return

        md5hash = hashlib.md5()
        with open(target_file, 'rb') as f:
            while True:
                data = f.read(1024)
                if len(data) == 0:
                    break
                md5hash.update(data)
        calc_md5 = md5hash.hexdigest()
        if calc_md5 != self.md5:
            raise RuntimeError(
                'File {target_file} has wrong hash (expected {self.md5}, seen {calc_md5})!'.format(**locals()))

    def extract(self):
        if self.build_dir:
            return

        self.log.info('Extracting %s', self.file)
        sp.check_call(['tar', 'xf', self.target_file, '-C', BUILD_DIR])

    def run(self, cmd, cwd=None):
        path = os.getenv('PATH')
        env = {
            'LDFLAGS': '-L%s/lib' % self.target_dir,
            'LD_LIBRARY_PATH': '%s/lib' % self.target_dir,
            'PKG_CONFIG_PATH': '%s/lib/pkgconfig' % self.target_dir,
            'CFLAGS': '-I%s/include' % self.target_dir,
            'PATH': '%s/bin:%s' % (self.target_dir, path)
        }

        self.log.debug('Running: %s %s',
                       ' '.join(k + '=' + v for k, v in env.items()),
                       ' '.join('"' + x + '"' if ' ' in x else x for x in cmd))

        with ProcessLog(self, logger=logging.getLogger("run")) as pl:
            sp.check_call(cmd,
                          stdout=pl.stdout,
                          stderr=sp.STDOUT,
                          cwd=cwd if cwd else self.build_dir,
                          env=env)

    def build(self, set_info, task_id):
        fname = self.build_dir + '/.built'
        if os.path.exists(fname):
            self.log.info('Target %s has already been built', self.file)
            return

        self.target_dir = os.path.abspath(TARGET_DIR)
        self.task_id = task_id

        set_info('Building %s' % self.file)
        self.configure()
        self.install()
        with open(fname, 'w'):
            pass

    def configure(self):
        self.run(['./configure', '--prefix=' + self.target_dir] + self.configure_args)

    def install(self):
        self.run(['make', '-j', CPU_COUNT])
        self.run(['make', '-j', CPU_COUNT, 'install'])


class OpenSSLBuildable(Buildable):
    def configure(self):
        self.run(['./config', '--prefix=' + self.target_dir])


class CmakeBuildable(Buildable):
    def __init__(self,
                 name,
                 url,
                 md5,
                 file='',
                 configure_args=None,
                 cmake_args=None,
                 pre_cmake=None):
        super(CmakeBuildable, self).__init__(name, url, md5, file, configure_args)
        self.cmake_args = cmake_args
        self.pre_cmake = pre_cmake

    def configure(self):
        if self.pre_cmake:
            self.run(self.pre_cmake)
        self.run(['cmake', '-G', 'Unix Makefiles', '-DCMAKE_INSTALL_PREFIX=' + self.target_dir] + self.cmake_args)


class X265Buildable(Buildable):
    def configure(self):
        args = [
            '-DCMAKE_INSTALL_PREFIX=' + self.target_dir,
            '-DLINKED_10BIT=ON',
            '-DLINKED_12BIT=ON',
            '-DEXTRA_LINK_FLAGS=-L.',
            '-DEXTRA_LIB=x265_main10.a;x265_main12.a',
        ]
        high_bit_args = [
            '-DHIGH_BIT_DEPTH=ON',
            '-DEXPORT_C_API=OFF',
            '-DENABLE_SHARED=OFF',
            '-DENABLE_CLI=OFF',
        ]

        for n in [8, 10, 12]:
            try:
                os.makedirs(self.build_dir + '/' + str(n) + 'bit')
            except:
                pass

        self.run(['cmake', '../source', '-DENABLE_HDR10_PLUS=ON'] + high_bit_args, cwd=self.build_dir + '/10bit')
        self.run(['cmake', '../source', '-DMAIN12=ON'] + high_bit_args, cwd=self.build_dir + '/12bit')
        self.run(['cmake', '../source'] + args, cwd=self.build_dir + '/8bit')
        self.run([self._sed, '-i', 's/-lgcc_s/-lgcc_eh/g', 'x265.pc'], cwd=self.build_dir + '/8bit')

    def install(self):
        self.run(['make', '-j', CPU_COUNT], cwd=self.build_dir + '/10bit')
        shutil.move(self.build_dir + '/10bit/libx265.a', self.build_dir + '/8bit/libx265_main10.a')

        self.run(['make', '-j', CPU_COUNT], cwd=self.build_dir + '/12bit')
        shutil.move(self.build_dir + '/12bit/libx265.a', self.build_dir + '/8bit/libx265_main12.a')

        self.run(['make', '-j', CPU_COUNT], cwd=self.build_dir + '/8bit')
        shutil.move(self.build_dir + '/8bit/libx265.a', self.build_dir + '/8bit/libx265_main.a')

        if platform.system() == 'Darwin':
            self.run(['libtool', '-static', '-o', 'libx265.a', 'libx265_main.a',
                      'libx265_main10.a', 'libx265_main12.a'], cwd=self.build_dir + '/8bit')
        else:
            script = 'CREATE libx265.a\n' + \
                     'ADDLIB libx265_main.a\n' + \
                     'ADDLIB libx265_main10.a\n' + \
                     'ADDLIB libx265_main12.a\n' + \
                     'SAVE\n' + \
                     'END\n'
            self.run(['ar', '-M', script], cwd=self.build_dir + '/8bit')

        self.run(['make', 'install'], cwd=self.build_dir + '/8bit')


class AutogenBuildable(Buildable):
    def configure(self):
        self.run(['./autogen.sh'])
        super(AutogenBuildable, self).configure()


class ReconfBuildable(Buildable):
    def configure(self):
        self.run(['autoreconf', '-fiv'])
        super(ReconfBuildable, self).configure()


class Patch(Buildable):
    def build(self, set_info, task_id):
        fname = self.file + '.built'
        if os.path.exists(fname):
            self.log.info('Target %s has already been built', self.file)
            return
        set_info('Patching %s' % self.file)
        self.run(['patch', '-Np1', '-i', os.path.abspath(self.target_file)])
        with open(fname, 'w'):
            pass

    def extract(self):
        pass


class VorbisBuildable(AutogenBuildable):
    def configure(self):
        super(VorbisBuildable, self).configure()
        self.run([self._sed, '-i', r's/Requires.private/Requires/', 'vorbis.pc'])
        self.run([self._sed, '-i', r's/Requires.private/Requires/', 'vorbisenc.pc'])
        self.run([self._sed, '-i', r's/Requires.private/Requires/', 'vorbisfile.pc'])


class AssBuildable(Buildable):
    def configure(self):
        super(AssBuildable, self).configure()
        self.run([self._sed, '-i', r's/Requires.private/Requires/', 'libass.pc'])


class ZimgBuildable(AutogenBuildable):
    def configure(self):
        super(ZimgBuildable, self).configure()
        self.run([self._sed, '-i', r'/Libs.private/d; /Libs/s/lzimg/lzimg -lstdc++/', 'zimg.pc'])


class TheoraBuildable(Buildable):
    def configure(self):
        super(TheoraBuildable, self).configure()
        self.run([self._sed, '-i', r's/png_sizeof/sizeof/', 'examples/png2theora.c'])


class FfmpegBuildable(Buildable):
    def configure(self):
        super(FfmpegBuildable, self).configure()
        self.run([self._sed,
                  '-i',
                  '-r',
                  r's/FFMPEG_CONFIGURATION "(.*)"/FFMPEG_CONFIGURATION "[sb:\/\/%s] \1"/' % self.task_id,
                  'config.h'])
