# coding=utf-8

import os

import sandbox.projects.common.constants as consts
import sandbox.common.types.misc as ctm

from sandbox.sandboxsdk import parameters, process
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sdk2 import ssh
from sandbox.projects.common.build import ArcadiaTask
from sandbox.projects.common import gnupg, debpkg


class MusicDebRelease(ArcadiaTask.ArcadiaTask):
    """
    Music Backend debian packages build and dupload
    """

    # This parameter will be available in "Advanced" section of the task parameters
    class ContainerParameter(parameters.Container):
        name = consts.SANDBOX_CONTAINER
        description = 'Container the task should execute in'
        default_value = '550066464'  # trusty default
        required = False

    # ArcadiaTask uses sdk1 :(
    type = 'MUSIC_DEB_RELEASE'
    privileged = True
    input_parameters = ArcadiaTask.ArcadiaTask.input_parameters + [ContainerParameter]
    dns = ctm.DnsType.DNS64
    execution_space = 60 * 1024

    registry_token = None

    DIST_REPO = 'media-java'
    DUPLOAD_CONF = {
        DIST_REPO: {
            'fqdn': "media-java.dupload.dist.yandex.ru",
            'method': "scpb",
            'login': "robot-teamcity-music",
            'incoming': "/repo/media-java/mini-dinstall/incoming/",
            'dinstall_runs': 1,
        }
    }
    package_version_str = None
    arcadia_src_dir = None

    def on_prepare(self):
        registry_token = self.get_vault_data("MUSIC", "robot-teamcity-music-registry")
        assert registry_token, 'Registry Oauth token must not be None or False'
        self.registry_token = registry_token

        arcadia = self.get_arcadia_src_dir()
        self.arcadia_src_dir = arcadia
        revision = Arcadia.get_revision(arcadia)
        music = os.path.join(arcadia, 'music')
        os.chdir(music)

        version_p = process.run_process(
            'deb-scripts/deb-utils.sh version',
            shell=True,
            outs_to_pipe=True,
            wait=True,
            check=True
        )
        version, _ = version_p.communicate()
        version_str = '{}.{}'.format(version.strip('\n'), revision)
        self.ctx['version_str'] = version_str
        self.package_version_str = version_str
        process.run_process(
            'sed "s/#EXACT_CURRENT_VERSION#/%s/g" debian/control.template > debian/control' % version_str,
            shell=True,
            wait=True,
            check=True
        )

    def do_execute(self):
        os.makedirs("up")
        os.makedirs("work")
        os.makedirs("arcadia_rw")
        process.run_process(
            'mount -t overlay -o lowerdir={},upperdir=up,workdir=work none arcadia_rw'.format(self.arcadia_src_dir),
            wait=True,
            check=False,
            log_prefix='mount'
        )
        process.run_process('mount', wait=True, check=False, log_prefix='mount')
        os.chdir("arcadia_rw/music")

        self.debuild_and_dupload()
        self.build_docker_image()

        os.chdir("../../")
        process.run_process('pkill dockerd', wait=True, check=False, log_prefix='cleanup')
        process.run_process('umount arcadia_rw', wait=True, check=False, log_prefix='cleanup')
        process.run_process('mount', wait=True, check=False, log_prefix='cleanup')
        process.run_process('rm -vfr work', wait=True, check=False, log_prefix='cleanup')

    def debuild_and_dupload(self):
        environment = os.environ.copy()
        environment['DEBFULLNAME'] = 'Sandbox'
        environment['EMAIL'] = 'robot-teamcity-music@yandex-team.ru'
        #  "robot-teamcity-music-gpg-private", "robot-teamcity-music-gpg-public" - имена
        #  записей в Vault, в которых сохранены base64 представления секретного
        #  и публичного GPG-ключей
        #  "MUSIC" - имя владельца записей в Vault
        with gnupg.GpgKey(
            self,
            "MUSIC",
            "robot-teamcity-music-gpg-private",
            "robot-teamcity-music-gpg-public"
        ):
            with self.current_action('Initializing...'):
                process.run_process(
                    'dch -v "%s" -b "incver to build debian package"' % self.package_version_str,
                    wait=True,
                    check=False,
                    log_prefix='dch',
                    environment=environment
                )
                process.run_process(
                    'dch -r unstable',
                    wait=True,
                    check=False,
                    log_prefix='dch',
                    environment=environment
                )
            with self.current_action('Building debian packages...'):
                process.run_process(
                    'debuild -krobot-teamcity-music@yandex-team.ru -b',
                    wait=True,
                    check=True,
                    log_prefix='debuild'
                )

            with ssh.Key(self, "MUSIC", "robot-teamcity-music-ssh"):
                with self.current_action('Uploading debian packages...'):
                    with debpkg.DebRelease(self.DUPLOAD_CONF) as deb:
                        deb.debrelease(['--to', self.DIST_REPO])

    def build_docker_image(self):
        with self.current_action('Building docker images...'):
            process.run_process(
                './docker/build.sh -v="%s" --push --sandbox' % self.package_version_str,
                wait=True,
                check=True,
                log_prefix='docker-build',
                environment={'REGISTRY_TOKEN': self.registry_token},
            )
