import logging
from dataclasses import dataclass
from datetime import datetime, timedelta
from functools import partial

import requests

log = logging.getLogger(__name__)


@dataclass(init=False)
class LogRecord:
    time: datetime
    login: str
    service: str
    project: str
    status: str


class Juggler:
    api = "https://juggler-api.search.yandex.net"

    def __init__(self, token):
        self.token = token
        self.session = self.prep_session()

    def run(self):
        self.get_escalations_log()

    def prep_session(self):
        s = requests.Session()
        s.headers["Authorization"] = f"OAuth {self.token}"
        return s

    def get_escalations_log(self, projects: list, days: int):
        url = f"{self.api}/v2/escalations/get_escalations_log"

        body = {
            "filters": [
                {"project": p} for p in projects
            ],
            "only_running": False,
            "page_size": 100,  # max allowed size
        }

        req = requests.Request("POST", url=url, json=body)
        prepped = self.session.prepare_request(req)

        try:
            res = self.session.send(prepped, verify=False)
            res.raise_for_status()
        except:
            log.exception("Something went wrong", exc_info=True)
            raise

        esc = res.json()['escalations']
        log.info("Received {} escalations from juggler".format(len(esc)))
        day_filter = partial(_filter_escalation, days_ago=days)
        esc = filter(day_filter, esc)

        calls: list[LogRecord] = []
        for e in esc:
            for call_log in e["log"]:
                if not call_log['calls']:
                    continue

                for call in call_log['calls']:
                    record = LogRecord()
                    record.time = datetime.fromtimestamp(call["start_time"])
                    record.status = call["status"]

                    record.project = e['project']
                    record.service = e['service']

                    if "simple_login" in call_log:
                        record.login = call_log["simple_login"]
                    elif "abc_duty_login" in call_log:
                        record.login = call_log["abc_duty_login"]["value"]
                    else:
                        raise Exception("No caller login found for escalation: {}".format(e))

                    if not record.project:
                        record.project = "not_set"
                    calls.append(record)
        return calls


def _filter_escalation(escalation: dict, days_ago: int):
    now = datetime.now()
    yesterday = now - timedelta(days=1)
    days_ago = now - timedelta(days=days_ago)

    start_time = datetime.fromtimestamp(escalation["start_time"])
    if days_ago < start_time < yesterday:
        if escalation['log']:
            return True
        return False
    return False
