import re

import yaml
from enum import Enum


class Reducer(Enum):
    PERCENTILE = 'percentile'
    COUNT = 'count'


class Actions(Enum):
    SUBTRACT = '-'
    DIVIDE = '/'


ALL_ACTIONS = set([x.value for x in Actions])


class FieldMapper:
    def __init__(self, text):
        self._tokens = re.split(r'\s+', text)

    def __str__(self):
        return ' '.join(self._tokens)

    def __getitem__(self, i):
        return self._tokens[i]

    def __eq__(self, rhs):
        return self._tokens == rhs

    def __ne__(self, rhs):
        return self._tokens != rhs

    @property
    def all_fields(self):
        return self._tokens

    @property
    def fields(self):
        return [x for x in self._tokens if x not in ALL_ACTIONS]


class EventConfig:
    def __init__(self, struct):
        self._target_name = struct['targetName']
        self._event_mapper = FieldMapper(struct['eventMapper'])
        self._extract_mapper = FieldMapper(struct['extractMapper'] if 'extractMapper' in struct else '')
        if struct['reducer'] == Reducer.PERCENTILE.value:
            self._reducer = Reducer.PERCENTILE
        elif struct['reducer'] == Reducer.COUNT.value:
            self._reducer = Reducer.COUNT
        else:
            raise RuntimeError('Unknown reducer: ' + struct['reducer'])

    def __repr__(self):
        return 'Event {%s <- %s(%s @ %s)}' % (
            self.target_name, self._reducer.value, self.event_mapper, self.extract_mapper)

    @property
    def target_name(self):
        return self._target_name

    @property
    def event_mapper(self):
        return self._event_mapper

    @property
    def extract_mapper(self):
        return self._extract_mapper

    @property
    def reducer(self):
        return self._reducer


class Config:
    def __init__(self, config_text):
        data = yaml.safe_load(config_text)

        self._percentile = [float(x) for x in data['percentile']]
        self._events = [EventConfig(x) for x in data['events']]

    def __str__(self):
        evt_str = '\n    '.join([str(x) for x in self.events])
        return 'CONFIG {\n  percentile: %s\n  events: [\n    %s\n  ]\n}' % (self.percentile, evt_str)

    @property
    def percentile(self):
        return self._percentile

    @property
    def events(self):
        return self._events
