import json
from collections import defaultdict
from datetime import datetime, timedelta
from time import time

import requests
from enum import Enum

from Config import Config
from SQLGenerator import SQLGenerator


class QueryStatus(Enum):
    RUNNING = 'RUNNING'
    COMPLETED = 'COMPLETED'
    OTHER = 'OTHER'


class Processor:
    yql_url = "https://yql.yandex-team.ru/api/v2/operations"
    solomon_url = "https://solomon.yandex.net/api/v2"

    project = 'music'
    cluster = 'mobile'
    service = 'mobile'

    def __init__(self, config_text, token, is_fake):
        self._session = requests.Session()
        self._session.headers.update({
            'Authorization': 'OAuth ' + token,
            'Content-type': 'application/json',
            'Accept': 'application/json',
        })

        self._config = Config(config_text)
        self._sql_generator = SQLGenerator(self._config, "fake" if is_fake else self._solomon_find_start_ts())

    def start_processing_logs(self):
        sql = self._sql_generator.sql
        return self._query_run(sql)

    def finalize_processing_logs(self, id_):
        data = self._query_data(id_, len(self._sql_generator.tables))

        # here we have data in groups. let us regroup it into one set
        grouper = defaultdict(dict)
        for table in data:
            for row in table:
                grouper[row['data_table_name']].update(row)
        data = list(grouper.values())

        # data is grouped but there are values that read like 'NaN', get rid of those
        for table in data:
            for k, v in list(table.items()):
                if v == 'NaN':
                    del table[k]

        return self._solomon_send(data)

    def _solomon_find_start_ts(self):
        now = time()
        then = now - 86400

        data = {
            "downsampling": {
                "maxPoints": 100
            },
            'from': int(then * 1000),
            'to': int(now * 1000),
            "program": "{cluster='mobile', service='mobile', sensor='*'}",
        }
        r = self._session.post('{api}/projects/{p}/sensors/data'.format(p=self.project, api=self.solomon_url),
                               json=data)
        r.raise_for_status()
        result = r.json()

        for ts in result['vector']:
            if 'timeseries' in ts and 'timestamps' in ts['timeseries']:
                return datetime.fromtimestamp(ts['timeseries']['timestamps'][-1] // 1000).isoformat()

    def _query_run(self, query):
        result = self._session.post(
            url=self.yql_url,
            json={
                'content': query,
                'action': 'RUN',
                'type': 'SQLv1',
            })
        result.raise_for_status()
        return result.json()['id']

    def query_status(self, id_):
        result = self._session.get(self.yql_url + '/' + id_ + '/meta')
        print(result.json())
        status = result.json()['status']
        if status == 'RUNNING':
            return QueryStatus.RUNNING
        elif status == 'COMPLETED':
            return QueryStatus.COMPLETED
        else:
            return QueryStatus.OTHER

    def _query_data(self, id_, index_count):
        result = []
        for i in range(index_count):
            tmp = self._session.get('%s/%s/results_data?format=json&write_index=%s' % (self.yql_url, id_, i))
            result.append([json.loads(s) for s in tmp.text.split()])
        return result

    def _solomon_send(self, data):
        metrics = []

        for event in data:
            dt = datetime.strptime(event['data_table_name'], '%Y-%m-%dT%H:%M:%S') - timedelta(hours=3)
            ts = dt.isoformat() + 'Z'
            del event['data_table_name']

            for key, value in event.items():
                metrics.append({
                    'labels': {
                        'sensor': key,
                    },
                    'ts': ts,
                    'value': value,
                })

        result = {
            'metrics': metrics
        }

        resp = self._session.post('{api}/push?project={p}&cluster={c}&service={s}'
                                  .format(p=self.project, s=self.service, c=self.cluster, api=self.solomon_url),
                                  json=result)
        resp.raise_for_status()
        return resp.text
