# -*- coding: UTF-8 -*-

import json
import logging
import os
import shutil
import stat

import sandbox.common.errors as ce
from sandbox import sdk2
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog


class MusicRestoreYdb(MusicBaseTask, sdk2.Task):
    """Creates ydb backup"""

    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        database = sdk2.parameters.String(
            "YDB database id",
            description="ID of database (not YDB name)",
            required=True,
        )

        check_interval_time = sdk2.parameters.Integer(
            "Check progress status interval time (sec.)",
            default=300,
            required=False,
        )

        cloud_name = sdk2.parameters.String(
            "cloud name",
            default="musicbackend",
            required=True,
        )

        yc_token_name = sdk2.parameters.String(
            "yc token name",
            default_value='yc_token',
            description="Token name to extract from YAV, "
                        "necessary if you want to export an environment different from PROD"
        )

        token = sdk2.parameters.YavSecret(
            "YAV secret with ydb-yt migrations and backups",
            default='sec-01ej69jtmv3t8g2675y81dc5bv',
            required=True
        )

        restore_prefix = sdk2.parameters.String(
            "Prefix to restore to",
            default='/'
        )

        backup_id = sdk2.parameters.String(
            "Backup id to restore",
            required=True
        )

        yc_binary = sdk2.parameters.Resource(
            "yc binary"
        )

    def __execute_yc_binary(self, args):
        with ProcessLog(self, logger="yc_execution") as pl:
            logging.info("Executing: [yc " + ' '.join(args) + ']')
            args = ['./yc'] + args
            process = sp.Popen(args, stdout=sp.PIPE, stderr=pl.stderr)
            outs = process.communicate()
            logging.info(
                'YC cli call returned {exitcode}.\nStdout:{output}\nStderr:{outerr}'.format(exitcode=process.returncode,
                                                                                            output=outs[0],
                                                                                            outerr=outs[1])
            )
            if process.returncode != 0:
                raise ce.TaskError('YC cli call failed.\nStderr: {output}'.format(output=outs[1]))
            return outs[0]

    def on_execute(self):
        token = self.Parameters.token.data()[self.Parameters.yc_token_name]
        resource = sdk2.Resource[self.Parameters.yc_binary]
        resource_data = sdk2.ResourceData(resource)
        executable_path = './yc'
        shutil.copyfile(str(resource_data.path), executable_path)
        # sometimes sb loses attributes
        os.chmod(
            executable_path, os.stat(executable_path).st_mode | stat.S_IEXEC
        )

        # log version just in case
        self.__execute_yc_binary(["-v"])
        self.__execute_yc_binary(["config", "profile", "create", "main"])
        self.__execute_yc_binary(["config", "set", "cloud-id", self.Parameters.cloud_name])
        self.__execute_yc_binary(["config", "set", "token", token])
        with self.memoize_stage.start_restore:
            res = json.loads(
                self.__execute_yc_binary(
                    ["--format", "json", "ydb", "database", "restore"] +
                    ["--id", self.Parameters.database] +
                    ["--backup-id", self.Parameters.backup_id] +
                    ["--target-path", self.Parameters.restore_prefix] +
                    ["--async"]
                )
            )
            self.Context.restore_id = res['id']
        if not self.Context.operation_done:
            res = json.loads(
                self.__execute_yc_binary(["--format", "json", "operation", "get", self.Context.restore_id])
            )
            if 'error' in res:
                raise ce.TaskError('Restore failed')
            if 'done' not in res:
                raise sdk2.WaitTime(self.Parameters.check_interval_time)
            self.Context.operation_done = True
        logging.info("Restore done")
