# -*- coding: utf-8 -*-
from __future__ import unicode_literals
from email.mime.text import MIMEText
import requests
from sandbox import sdk2
from sandbox.common import errors
from sandbox.sandboxsdk import environments
import logging
import smtplib


class MusicSendEmailWithMusicNetworks(sdk2.Task):
    """
    Collects ip addr and networks. Sends them as email
    """
    racktables_api = "https://racktables.yandex.net/export"
    nets_api = racktables_api + "/nets-by-project.php"
    ipv4_nets = racktables_api + "/allipv4nets.php"
    hbf_api = "https://hbf.yandex.net/"
    mds_tun64_args = {
        "format": "allipnets",
        "text": "{TUN64 block} and {$vlan_1302}",
    }

    nets = set()
    ips = set()

    domains = (
        "api.music.yandex.net",
        "api.music.yandex.ru",
        "ws-api.music.yandex.net",
        "avatars.mds.yandex.net",
        "avatars.yandex.net",
        "match.music.yandex.net",
        "mobileproxy-certificate.passport.yandex.net",
        "music.yandex.ru",
        "social.yandex.ru",
        "storage.mds.yandex.net",
        "yapic.yandex.ru",
        "report.appmetrica.yandex.net",
        "startup-slb.mobile.yandex.net",
        "rosenberg.appmetrica.yandex.net",
        "redirect.appmetrica.yandex.com",
        "push.api.appmetrica.yandex.net",
        "cold.strm.yandex.net",
        "strm.yandex.ru",
        "strm.yandex.net",
    )
    mail_body = "Список ip адресов для обнуления трафика\n\n"
    mail_from = "Yandex Music <media-admin@yandex-team.ru>"
    mail_to = [
        "PS_group_kz@tele2.kz",
        "st-tele_kz@tele2.kz",
        "ALHarutyunyan@beeline.am",
        "ANMikayelyan@beeline.am",
        "vovagor@yandex-team.ru",
        "erzaev@beeline.uz",
        "RKhismatullin@beeline.uz",
        "SNasimov@beeline.uz",
        "td@sberbank-tele.com",
        "vselivanov@beeline.kz",
        "AShukenov@beeline.kz",
        "TZarypov@beeline.uz",
        "AnAKim@beeline.uz",
    ]

    mail_subj = "Список ip адресов Яндекс Музыки"

    class Parameters(sdk2.Task.Parameters):
        description = "Collects ip addr and networks. Sends them as email"
        debug = sdk2.parameters.Bool("Turn on debug mode", description="Sends message to stepanar@", default=False)

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment("dnspython"),
        ]
        cores = 1
        ram = 1024

    @staticmethod
    def get_addr(hostname, nets):
        """
        Add  addr of a hostname to nets list
        """
        from dns import resolver, rdatatype
        ipvers = [rdatatype.A, rdatatype.AAAA]

        res = resolver.Resolver()
        res.nameservers = ['2a02:6b8::1']  # ns1.yandex.net

        try:
            for ipver in ipvers:
                answer = res.query(hostname, rdtype=ipver)
                for record in answer:
                    nets.add(record.to_text())

        except Exception as ex:
            raise errors.TaskError(ex)

    @staticmethod
    def get_networks(api, nets):
        """
        Get _MUSICNETS_ and _ELLIPTICSNETS_ ipv4  and ipv6 networks from hbf
        """
        macros = ["_MUSICNETS_",
                  "_ELLIPTICSNETS_",
                  "_MDS_STORAGE_NETS_",
                  "_MDS_STORAGE_TUN64_",
                  "_STRMNETS_",
                  "_REGIONALCACHINGNETS_",
                  "_CDNFOREIGNNETS_",
                  "_CDNNETS_",
                  ]
        try:
            for macro in macros:
                request = requests.get(api + "macros/" + macro + "?format=text")
                networks = request.text
                networks = networks.splitlines()
                for line in networks:
                    if "@" in line:
                        # cut project id from network
                        # input  -> "671@2a02:6b8:c00::/40"
                        # output -> "2a02:6b8:c00::/40"
                        line = line.split("@")[1]
                    nets.add(line)
            return nets
        except requests.RequestException as exc:
            raise errors.TaskError("Racktables api: {}".format(exc))

    @staticmethod
    def get_mds_tun(api, payload, nets):
        """
        Get MDS tun64 nets from racktables
        """
        try:
            request = requests.get(api, params=payload)
            networks = request.text
            networks = networks.splitlines()
            for line in networks:
                nets.add(line)
            return nets
        except requests.RequestException as exc:
            raise errors.TaskError("Racktables api: {}".format(exc))

    @staticmethod
    def create_mail(mail_to, mail_from, body, subj):
        """
        Create an email object
        """
        msg = MIMEText(body, 'plain', 'utf-8')
        msg["From"] = mail_from
        msg["Subject"] = subj.encode("utf-8")
        return msg

    def on_execute(self):
        self.get_networks(self.hbf_api, self.nets)
        self.get_mds_tun(self.ipv4_nets, self.mds_tun64_args, self.nets)
        for domain in self.domains:
            self.get_addr(domain, self.ips)
        all = sorted(self.ips | self.nets)
        for net in all:
            self.mail_body += net + "\n"
        message = self.create_mail(self.mail_to, self.mail_from, self.mail_body, self.mail_subj)
        logging.info(message)
        if self.Parameters.debug:
            self.mail_to = "stepanar@yandex-team.ru"
        try:
            server = smtplib.SMTP("yabacks.yandex.ru")
            server.sendmail(self.mail_from, self.mail_to, message.as_string())
            server.quit()
        except smtplib.SMTPException as exc:
            raise errors.TaskError("Didn't send the mail {}".format(exc))
