import urllib2
import json
import re
import os
import urlparse
import logging


import sandbox.common.types.client as ctc
import sandbox.common.errors as errors

import sandbox.sandboxsdk.parameters as sb_parameters
import sandbox.sandboxsdk.errors as sb_errors
import sandbox.sdk2.vcs.svn as svn

import sandbox.projects.common.arcadia.sdk as arcadia_sdk
from sandbox.projects.common.build import YaPackage
import sandbox.projects.common.build.parameters as build_parameters
import sandbox.projects.common.constants as consts
import sandbox.projects.common.utils as utils


class BranchShortnameParam(sb_parameters.SandboxStringParameter):
    name = 'branch_shortname'
    description = 'Release branch short name (i.e. release version)'
    default_value = ''


class RevisionParam(sb_parameters.SandboxIntegerParameter):
    name = 'rev'
    description = 'Release branch revision (empty for HEAD)'
    default_value = None


class ReleaseMusic(YaPackage.YaPackage):
    type = 'RELEASE_MUSIC'

    environment = []
    client_tags = ctc.Tag.GENERIC
    cores = 17
    input_parameters = [BranchShortnameParam,
                        RevisionParam,

                        build_parameters.DoNotRemoveResources,

                        YaPackage.TestsBlock,
                        # Tests
                        YaPackage.RunTestsParameter,
                        YaPackage.RunLongTestsParameter,
                        YaPackage.IgnoreFailTestsParameter,

                        YaPackage.PublishingBlock,
                        # Publishing
                        YaPackage.ChangelogParameter,
                        YaPackage.PublishPackageParameter,
                        YaPackage.KeyUserParameter,
                        YaPackage.MultiplePublishParameter,
                        YaPackage.PublishToParameter,
                        YaPackage.MultiplePublishToParameter,
                        YaPackage.MultiplePublishMappingParameter,
                        ]

    PROJECT_PATH = 'devtools/experimental/music'  # XXX set to 'music'

    DEFAULT_CONTEXT = {
        build_parameters.BuildType.name: consts.RELEASE_BUILD_TYPE,
        YaPackage.PackagesParameter.name: os.path.join(PROJECT_PATH, 'common', 'pkg.json'),
        YaPackage.PackageTypeParameter.name: YaPackage.DEBIAN,
        build_parameters.BuildSystem.name: consts.SEMI_DISTBUILD_BUILD_SYSTEM,
        build_parameters.CheckoutModeParameter.name: consts.CHECKOUT_MODE_MANUAL,
    }

    def _populate_context(self):
        """Populate context using YaPackage defaults and our defaults."""
        input_parameters_names = set(parameter.name for parameter in self.input_parameters)
        for parameter in YaPackage.YaPackage.input_parameters:
            if parameter.name not in input_parameters_names or self.ctx.get(parameter.name, None) is None:
                self.ctx[parameter.name] = parameter().default_value
        self.ctx.update(self.DEFAULT_CONTEXT)
        # XXX replace junk/workfork with music
        self.ctx[build_parameters.ArcadiaUrl.name] = 'arcadia:/arc/branches/junk/workfork/{}/arcadia{}'.format(
            self.ctx[BranchShortnameParam.name],
            '@' + str(self.ctx[RevisionParam.name]) if self.ctx[RevisionParam.name] else ''
        )

    def on_enqueue(self):
        self._populate_context()
        YaPackage.YaPackage.on_enqueue(self)

    def get_arcadia(self):
        self.arcadia_src_dir = arcadia_sdk.do_clone(utils.get_or_default(self.ctx, build_parameters.ArcadiaUrl), self, True)
        arcadia_sdk.do_build(source_root=self.arcadia_src_dir, targets=[self.PROJECT_PATH], build_threads=0, checkout=True, build_system=consts.YMAKE_BUILD_SYSTEM, clear_build=False)
        logging.debug('CHECKOUT DONE')
        return self.arcadia_src_dir

    def on_execute(self):
        arcadia_url = utils.get_or_default(self.ctx, build_parameters.ArcadiaUrl)
        if urlparse.urlparse(svn.Arcadia.normalize_url(arcadia_url)).scheme == svn.Arcadia.ARCADIA_HG_SCHEME:
            raise errors.TaskFailure('Hg is not supported for now')

        arcadia_url_comp = svn.Arcadia.parse_url(arcadia_url)
        if 'branch' not in arcadia_url_comp._asdict() or not arcadia_url_comp.branch:
            raise errors.TaskFailure('Release should be done from branch')

        failure = None
        try:
            YaPackage.YaPackage.on_execute(self)
        except sb_errors.SandboxSubprocessError as failure:
            logging.warn('Exception is delayed: %s', repr(failure))
            pass

        req = urllib2.Request('https://c.yandex-team.ru/api/package_version/yandex-music-web')
        json_responce = json.loads(urllib2.urlopen(req, timeout=30).read())
        stable_version = json_responce['yandex-music-web']['stable']['version']
        match_revision = re.match(r'^(?P<branch>([0-9a-zA-Z-]+\.){3})(?P<revision>.+)$', stable_version)
        if not match_revision or 'revision' not in match_revision.groupdict() or 'branch' not in match_revision.groupdict():
            raise Exception('wrong stable version: {}'.format(stable_version))

        stable_revision = match_revision.group('revision')
        stable_branch = match_revision.group('branch')
        match_is_svn = re.match(r'^[0-9]+$', stable_revision)
        if not match_is_svn:
            stable_base_revision = '3369571'  # XXX seed Arcadia revision
        else:
            logging.info('Got stable release branch %s at %s', stable_branch, stable_revision)

            logging.debug('SEEK FOR BASE OF %s', stable_branch)
            stable_base_revision, stable_init_revision = self.get_base_for_branch(stable_branch)
        logging.info('Stable base revision is %s', stable_base_revision)

        release_branch = arcadia_url_comp.branch
        logging.debug('SEEK FOR BASE OF %s', release_branch)
        release_base_revision, release_init_revision = self.get_base_for_branch(release_branch)
        logging.info('Release branch base revision is %s', release_base_revision)

        logging.debug('LOG FOR TRUNK')
        svn_log = svn.Arcadia.log(svn.Arcadia.trunk_url(self.PROJECT_PATH), revision_from=stable_base_revision, revision_to=release_base_revision)
        logging.debug('LOG FOR TRUNK RECEIVED %s', len(svn_log))

        logging.debug('LOG FOR BRANCH')
        # here we know that self.arcadia_src_dir is a branch local directory
        svn_log.extend(svn.Arcadia.log(self.arcadia_src_dir, revision_from=release_init_revision, revision_to=arcadia_url_comp.revision or 'HEAD'))
        logging.debug('LOG FOR BRANCH IS %s', len(svn_log))
        logging.debug(u'\n'.join(map(lambda x: x['msg'], svn_log)))

        svn_log = map(lambda x: x['msg'], filter(lambda x: 'DEVTOOLS-' in x['msg'] or 'AMREL-' in x['msg'], svn_log))
        logging.debug('LOG ENTRIES WITH TICKETS %s', len(svn_log))

        tickets = map(lambda x: re.search('(?P<ticket>(DEVTOOLS|AMREL)-\d+)', x).group('ticket'), svn_log)

        token = self.get_vault_data('music-st-token')

        changelog, tickets_without_test_scope = [], []
        for ticket in sorted(list(set(tickets))):
            req = urllib2.Request('https://st-api.yandex-team.ru/v2/issues/{}'.format(ticket), headers={'Authorization': 'OAuth {}'.format(token)})
            json_response = json.loads(urllib2.urlopen(req, timeout=30).read())
            summary = u'https://st.yandex-team.ru/{:<11}    {:<12}  {:<7} {}'.format(
                ticket,
                json_response.get('assignee', {}).get('id', ''),
                json_response.get('testScope', ''),
                json_response.get('summary', '').replace('\n', '\t')
            )
            changelog.append(summary)
            if not json_response.get('testScope', ''):
                tickets_without_test_scope.append(summary)

        self.set_info(u'=== Changelog ===\n{}'.format('\n'.join(changelog)))
        self.set_info(u'=== Tickets with empty test scope ===\n{}'.format('\n'.join(tickets_without_test_scope)))

        if failure:
            raise failure

    def get_base_for_branch(self, branch, revision=None, seek_step=100):
        arcadia_url = svn.Arcadia.branch_url(branch)

        if revision:
            revision_to = revision
        else:
            info = svn.Arcadia.info(arcadia_url)
            revision_to = int(info['commit_revision'])
        revision_from = revision_to - seek_step

        while True:
            log = svn.Arcadia.log(arcadia_url, revision_from=revision_from, revision_to=revision_to, track_copy=True, stop_on_copy=True)
            base_revision, init_revison = next(
                ((copy_entry[1], log_entry['revision']) for log_entry in log for copy_entry in log_entry['copies'] if
                 copy_entry[0] == '/trunk'), (None, None))
            if base_revision:
                break
            revision_to = revision_from
            revision_from = revision_to - seek_step
        return base_revision, init_revison


__Task__ = ReleaseMusic
