# -*- coding: utf-8 -*-

import logging
import shutil
from inspect import getdoc

from sandbox import sdk2
from sandbox import common
import sandbox.common.types.task as ctt

from sandbox.common.types.task import ReleaseStatus
from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard
from sandbox.projects.common.nanny import nanny


class BaseMusicShardmapResource(sdk2.Resource):
    """
    Base resource type for music-related shardmaps
    """
    releasers = ['robot-muzsearch']
    release_subscribers = []
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    shardmap = True


class BaseUploadMusicShardmap(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Base task for various music-related shardmap uploading tasks. MUST NOT be created explicitly.
    """

    class Parameters(sdk2.Task.Parameters):
        release_description = sdk2.parameters.String('Release description', default='')
        remote_file_name = sdk2.parameters.String('Remote file name', default='')
        created_resource_name = sdk2.parameters.String('Created resource name', default='shardmap')

    class Requirements(sdk2.task.Requirements):
        disk_space = 200    # MB, shardmap file is really tiny

    class Context(sdk2.Context):
        do_release = False
        child_task_id = None

    def _create_resource(self, filename):
        return self.resource_class(
            self,
            '{} {}'.format(getdoc(self.resource_class), self.Parameters.release_description),
            filename
        )

    def on_execute(self):
        if self.Context.do_release:
            # Only executed in the subtask: wait for the parent task to complete, then release it
            with self.memoize_stage.wait_for_parent:
                raise sdk2.WaitTask(
                    tasks=[self.parent],
                    statuses=(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                    wait_all=True)
            if self.parent.status == ctt.Status.SUCCESS:
                self.server.release(
                    task_id=int(self.parent),
                    type=ReleaseStatus.STABLE,
                    subject='Release of a {}'.format(getdoc(self.resource_class)),
                    to='nglaz@yandex-team.ru')
            else:
                raise common.errors.TaskFailure('Parent task {} failed'.format(int(self.parent)))
        else:
            # Run a subtask to upload the shardmap, then copy it to the current task and release it from one more subtask
            with self.memoize_stage.upload_shardmap:
                logging.info('Creating a shardmap upload task...')
                params = {
                    'resource_type': self.resource_class.name,
                    'created_resource_name': self.Parameters.created_resource_name,
                    'remote_file_name': self.Parameters.remote_file_name,
                }
                upload_task = self.server.task({
                    'type': 'REMOTE_COPY_RESOURCE',
                    'context': params,
                    'children': True,
                })
                self.server.task[upload_task['id']].update({
                    'description': 'Upload of a {}'.format(getdoc(self.resource_class)),
                    'owner': self.owner,
                    'notifications': self.server.task[self.id].read()["notifications"],
                    'priority': {
                        'class': self.Parameters.priority.cls,
                        'subclass': self.Parameters.priority.scls,
                    },
                    'requirements': {
                        'disk_space': 1 << 27,  # 128 MB
                    }
                })
                self.server.batch.tasks.start.update([upload_task['id']])
                self.Context.child_task_id = upload_task['id']
                waited_statuses = set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
                raise sdk2.WaitTask([self.Context.child_task_id], waited_statuses, wait_all=True)

            # Check that the REMOTE_COPY_RESOURCE was successful
            subtasks = self.server.task.read(id=[self.Context.child_task_id], children=True, limit=1)["items"]
            logging.info("Subtasks: {}".format(subtasks))
            failed_subtasks = [subtask for subtask in subtasks if subtask["status"] not in ctt.Status.Group.SUCCEED]
            if failed_subtasks:
                raise common.errors.TaskFailure("Failed subtasks: {}".format(failed_subtasks))

            # Copy the shardmap resource from the subtask to release it within this task
            remote_resource = list(sdk2.Resource.find(task_id=self.Context.child_task_id, type=self.resource_class).limit(1))[0]
            remote_resource_data = sdk2.ResourceData(remote_resource)
            resource = self._create_resource(self.Parameters.created_resource_name)
            resource_data = sdk2.ResourceData(resource)

            logging.info('Copying resource from ' + str(remote_resource_data.path))
            shutil.copy(str(remote_resource_data.path), str(resource_data.path))
            resource_data.ready()

            # Create a subtask of the same type to trigger release of this current task
            logging.info("Self-releasing")
            type(self)(parent=self, do_release=True).enqueue()

    def __run_deployment_task(self):
        sub_ctx = {
            deploy_nanny_dashboard.ReleaseTask.name: self.id,
            deploy_nanny_dashboard.NannyDashboardName.name: self.nanny_dashboard_name,
            deploy_nanny_dashboard.NannyDashboardRecipeName.name: 'deploy_shardmap',
            deploy_nanny_dashboard.SandboxReleaseType.name: ReleaseStatus.STABLE,
            deploy_nanny_dashboard.WaitDeployment.name: 120,
            deploy_nanny_dashboard.NannyWaitDeployParameter.name: True,
            deploy_nanny_dashboard.GetServicesFromRecipe.name: True,
            deploy_nanny_dashboard.VaultOwner.name: 'robot-muzsearch',
            deploy_nanny_dashboard.VaultName.name: 'nanny_oauth_token',
        }
        sub_task_class = sdk2.Task[deploy_nanny_dashboard.DeployNannyDashboard.type]
        return sub_task_class(self, **sub_ctx).enqueue()

    def on_release(self, additional_parameters):
        self.__run_deployment_task()
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
