# coding: utf-8
import re
from textwrap import dedent

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.task import Status
from sandbox.projects.common import link_builder as lb
from sandbox.projects.music.deployment.helpers.AbcHelper import AbcHelper
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper


class MusicApiTest(sdk2.Task):
    _issue_re = re.compile(r'^(MUSICBACKEND|MEDIASEARCH)-\d+$')

    class Requirements(sdk2.Task.Requirements):
        environments = [TaskHelper.startrek_client_environment]

    class Parameters(sdk2.Task.Parameters):
        description = "Run api tests over test environment"

        arcadia_url = sdk2.parameters.String("The arcadia url",
                                             default="arcadia:/arc/trunk/arcadia",
                                             description=CONFIG.arcadia_description,
                                             required=True)

        arcadia_patch = sdk2.parameters.String("The patch file to apply", required=False)
        issue = sdk2.parameters.String("Startrek issue to inform to", required=False)  # type: str
        external_api_url = sdk2.parameters.String("Target external api url",
                                                  default="https://api.music.qa.yandex.net/",
                                                  description="default https://api.music.qa.yandex.net/, "
                                                              "e.g. https://api-pr-x.music.qa.yandex.net/",
                                                  required=False)
        internal_api_url = sdk2.parameters.String("Target internal api url",
                                                  default="http://music-web.music.qa.yandex.net/internal-api/",
                                                  description="default http://music-web.music.qa.yandex.net/internal-api/, "
                                                              "e.g. http://web-pr-x.music.qa.yandex.net/internal-api/",
                                                  required=False)

        abc_token = sdk2.parameters.YavSecret("ABC OAuth token secret", required=True,
                                              description="Yav secret with ABC tolen",
                                              default='sec-01fp7s63sd8dgjga0erprk7tfx')

        abc_token_key = sdk2.parameters.String("Key for abc token in yav secret", required=True,
                                               default="abc_token")

        env_var_secret = sdk2.parameters.YavSecret(
            "Secret with env var overrides",
            required=True,
            default='sec-01eafb2xdrt88xz4h6yk3t7bgn'
        )

        with sdk2.parameters.Output:
            allure_report = sdk2.parameters.String("Allure report url")

    def create_subtask(self):
        task_class = sdk2.Task["YA_MAKE_2"]

        test_params = "external_host={} internal_host={}" \
            .format(self.Parameters.external_api_url, self.Parameters.internal_api_url)

        arcadia_patch = "zipatch:" + str(self.Parameters.arcadia_patch) if self.Parameters.arcadia_patch else None

        sub_task = task_class(self,
                              __requirements__={"dns": ctm.DnsType.DNS64},
                              description="Test music http api",
                              checkout="True",
                              targets="music/backend/music-common/web-test-api-external",
                              test="True",
                              arcadia_patch=arcadia_patch,
                              test_log_level="debug",
                              test_failure_code="10",
                              test_retries="1",
                              test_params=test_params,
                              build_system="ya",
                              test_tag="ya:manual",
                              allure_report="True",
                              allure_report_ttl=21,
                              checkout_arcadia_from_url=str(self.Parameters.arcadia_url),
                              force_build_depends="True",
                              definition_flags="-DJDK_VERSION={}".format(CONFIG._DEFAULT_JDK_VERSION),
                              keep_on="True",
                              sandbox_tags="IPV6",
                              env_vars_secret=self.Parameters.env_var_secret)
        sub_task.enqueue()
        self.Context.subtask = sub_task.id
        self.Context.save()
        raise sdk2.WaitTask(sub_task.id, Status.Group.FINISH + Status.Group.BREAK)

    def check_subtask(self):
        token = sdk2.Vault.data(CONFIG.token)
        st = StartrekHelper(token)

        sub_task = sdk2.Task.find(id=self.Context.subtask).first()
        allure_report = sdk2.Resource["ALLURE_REPORT"].find(task=sub_task).first().http_proxy + "/index.html"

        msg = dedent(u'Subtask {subtask} status: {task_status}\n'
                     u'Allure report: <a href="{allure_report}">index.html</a>'
                     .format(subtask=lb.task_link(sub_task.id, plain=False),
                             task_status=sub_task.status,
                             allure_report=allure_report))

        self.set_info(info=msg, do_escape=False)

        self.Parameters.allure_report = allure_report

        if sub_task.status == Status.SUCCESS:
            if self.Parameters.issue:
                st.add_comment(self.Parameters.issue,
                               u'Тест api прошел успешно\n'
                               u'Sandbox task: ((https://sandbox.yandex-team.ru/task/{task_id} {task_id}))\n'
                               u'Allure report: (({allure_report} index.html))'
                               .format(task_id=self.id,
                                       allure_report=allure_report))
                st.patch_release_issue_statuses_raw(self.Parameters.issue, CONFIG.monitored_deployment_test_names,
                                                    self.__class__.__name__)
            return
        else:
            if self.Parameters.issue:

                th = TaskHelper()
                abc_token = self.Parameters.abc_token.data()[self.Parameters.abc_token_key]
                th.abc_helper = AbcHelper(abc_token)

                if self._issue_re.match(self.Parameters.issue):
                    summonee = st.ticket_assignee(self.Parameters.issue)
                else:
                    summonee = th.backend_on_duty()

                st.add_comment(self.Parameters.issue,
                               u'Тест api !!сломался!!\n'
                               u'Sandbox: ((https://sandbox.yandex-team.ru/task/{task_id} {task_id}))\n'
                               u'Allure report: (({allure_report} index.html))'
                               .format(task_id=self.id,
                                       allure_report=allure_report),
                               summonee=summonee)
            raise errors.TaskFailure()

    def on_execute(self):
        with self.memoize_stage.create_subtask:
            # will run only once
            self.create_subtask()

        self.check_subtask()
