from typing import Type, cast

import sandbox.sdk2 as sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.common import link_builder as lb
from sandbox.projects.music.deployment.helpers.BuildHelper import BuildHelper
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.projects.music.deployment.helpers.Nyan import nyan, MUSIC_BACKEND_CHAT_ID
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper


class MusicBuildJars(MusicBaseTask, TaskHelper):
    """ Build Music Jars based on the revision number """
    fail_on_any_error = True
    BUILD_DIR = 'build'
    OUT_DIR = 'out'

    class Requirements(cast(Type[sdk2.task.Requirements], sdk2.Task.Requirements)):
        disk_space = 75 * 1024  # in megabytes

    class Parameters(cast(Type[sdk2.task.Parameters], sdk2.Task.Parameters)):
        kill_timeout = 40 * 60  # TODO: reduce to 25 when only one JDK left
        description = "Build Yandex.Music JARs and configs"

        jdk_versions = sdk2.parameters.List("JDK Versions (numbers) to build code with",
                                            default=[CONFIG._DEFAULT_JDK_VERSION],
                                            description='Build code with specified jdk versions')

        url = sdk2.parameters.String("The arcadia url (or path on local sandbox)",
                                     description=CONFIG.arcadia_description,
                                     required=True)

        patch = sdk2.parameters.String("The patch file to apply")
        patch_over_revision = sdk2.parameters.Integer("The patch is from this revision")
        use_yt_cache = sdk2.parameters.Bool("Use YT cache", default=True)

        move_files = sdk2.parameters.Bool("Move build files instead of copying them", default=False)

        with sdk2.parameters.Output():
            branch = sdk2.parameters.String("The branch built from")
            revision = sdk2.parameters.Integer("The revision built from")

    def on_break(self, **kwargs):
        super(MusicBuildJars, self).on_break(**kwargs)
        message = "Sandbox task [{task_name}]({link}) failed\n".format(task_name=str(self.type),
                                                                       link=lb.task_link(self.id, plain=True))
        nyan(message, chat_id=MUSIC_BACKEND_CHAT_ID)

    def on_execute(self):
        self.check_authorization(self.author, CONFIG.token, CONFIG.auth_build)

        yt_token = sdk2.Vault.data(CONFIG.yt_token)
        arcanum_token = sdk2.Vault.data(CONFIG.arcanum_token)
        bh = BuildHelper(patch=str(self.Parameters.patch),
                         jdks=self.Parameters.jdk_versions,
                         set_info=self.set_info,
                         arcanum_token=str(arcanum_token),
                         use_yt_cache=bool(self.Parameters.use_yt_cache),
                         move_files=bool(self.Parameters.move_files))

        bh.run(self, str(self.Parameters.url), str(yt_token))

        self.Parameters.branch = cast(sdk2.parameters.String, bh.branch)
        self.Parameters.revision = cast(sdk2.parameters.String, bh.revision)

        self.set_info('Revision: <b>{}</b><br/>Branch: <b>{}</b>'
                      .format(self.Parameters.revision, self.Parameters.branch),
                      do_escape=False)
        self.hint('{}/{}/{}'.format(self.Parameters.branch, self.Parameters.revision, self.Parameters.patch))
