import os

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.projects.common.arcadia.sdk as asdk
import sandbox.sandboxsdk.environments as environments
import sandbox.sdk2 as sdk2
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
from sandbox.sdk2.helpers import subprocess, ProcessLog

from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper


class MusicNetLatencyResource(sdk2.Resource):
    """ Net Latency pinger """
    releasable = True


class Golang(environments.TarballToolkitBase):
    """ Golang with `go` tool embedded into it. """

    name = "Golang"
    resource_type = "GOLANG_PACKAGE"
    use_cache = True

    @property
    def bindir(self):
        """ Returns path to binary directory of the prepared environment. """
        return os.path.join(self.get_environment_folder(), "go", "bin")

    @property
    def gotool(self):
        """ Returns path to `go` executable of the prepared environment. """
        return os.path.join(self.bindir, "go")

    def prepare(self):
        basedir = super(Golang, self).prepare()
        self.update_os_path_env(os.path.join(basedir, "go", "bin"))
        return basedir


class MusicBuildNetLatency(ReleaseToNannyTask2, sdk2.Task, TaskHelper):
    """ Build NetLatency """

    class Requirements(sdk2.Task.Requirements):
        environments = (Golang('1.11'),)
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_PRECISE

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 2 * 60
        description = "Build NetLatency"
        fail_on_any_error = True

        url = sdk2.parameters.String("The arcadia url",
                                     default=CONFIG.arcadia_trunk,
                                     description=CONFIG.arcadia_description,
                                     required=True)

    def build(self, arc_dir):
        source_dir = os.path.join(arc_dir, 'music', 'tools', 'netlatency')

        cwd_dir = os.getcwd()
        result_path = os.path.join(cwd_dir, 'netlatency')
        os.mkdir(result_path)

        gopath_dir = os.path.join(cwd_dir, 'gopath')
        os.mkdir(gopath_dir)

        gocache_dir = os.path.join(cwd_dir, 'gocache')
        os.mkdir(gocache_dir)

        gotool = self.Requirements.environments[0].gotool

        text = 'Building netlatency<br><br>' + \
               '<b>source dir</b>: {}<br>'.format(source_dir) + \
               '<b>result path</b>: {}<br>'.format(result_path) + \
               '<b>gopath dir</b>: {}<br>'.format(gopath_dir) + \
               '<b>gocache dir</b>: {}<br>'.format(gocache_dir)
        self.set_info(text, do_escape=False)

        env = os.environ.copy()
        env['GOPATH'] = gopath_dir
        env['GO111MODULE'] = 'on'
        env['GOCACHE'] = gocache_dir

        try:
            with ProcessLog(self, logger="go_env") as pl:
                subprocess.check_call([gotool, 'env'],
                                      stdout=pl.stdout,
                                      stderr=subprocess.STDOUT,
                                      env=env)

            with ProcessLog(self, logger="go_build_server") as pl:
                subprocess.check_call([gotool, 'build', '-o', os.path.join(result_path, 'server'), 'main/server.go'],
                                      stdout=pl.stdout,
                                      stderr=subprocess.STDOUT,
                                      cwd=source_dir,
                                      env=env)

            with ProcessLog(self, logger="go_build_client") as pl:
                subprocess.check_call([gotool, 'build', '-o', os.path.join(result_path, 'client'), 'main/client.go'],
                                      stdout=pl.stdout,
                                      stderr=subprocess.STDOUT,
                                      cwd=source_dir,
                                      env=env)
        finally:
            # the cleanup is necessary because of grants
            with ProcessLog(self, logger="cleanup") as pl:
                clean_command = "find {0} -type d -print0 | xargs -0 chmod 775;" + \
                                "find {0} -type f -delete"
                subprocess.check_call(clean_command.format(gopath_dir),
                                      shell=True, stdout=pl.stdout, stderr=subprocess.STDOUT)
                subprocess.check_call(clean_command.format(gocache_dir),
                                      shell=True, stdout=pl.stdout, stderr=subprocess.STDOUT)

        return result_path

    def on_execute(self):
        with asdk.mount_arc_path(self.Parameters.url) as arc_dir:
            result_path = self.build(arc_dir)

        resource = MusicNetLatencyResource(self, "NetLatency resource", result_path)
        data = sdk2.ResourceData(resource)
        data.ready()

        self.set_info('Successfully built the resource')
