import logging
import multiprocessing
import os
import platform
import re
import subprocess

import requests


CORES = multiprocessing.cpu_count()


class Source:
    def __init__(self, url):
        self._log = logging.getLogger("Source")

        self._url = url

        m = re.match(r'.*/(.*)(\.tar\.\w+)$', url)
        self._dir = m.group(1)
        self._file = m.group(1) + m.group(2)

    @property
    def file(self):
        return self._file

    @property
    def url(self):
        return self._url

    @property
    def dir(self):
        return self._dir

    @property
    def prefix(self):
        return os.path.abspath("./inst")

    def _download(self):
        self._log.info('Downloading %s into %s' % (self.url, self.file))
        r = requests.get(self.url, stream=True)
        with open(self.file, 'wb') as f:
            for chunk in r.iter_content(chunk_size=64000):
                f.write(chunk)

    def _run(self, cmd, shell=False):
        self._log.info('Running %s in %s' % (str(cmd), self.dir))
        proc = subprocess.Popen(cmd, cwd=self.dir, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, shell=shell)
        out = proc.communicate()
        self._log.info(out[0])
        if proc.returncode > 0:
            raise RuntimeError('Error code %s' % proc.returncode)
        return out[0]

    def prepare(self):
        if not os.path.exists(self.file):
            self._download()

        self._log.info('Extracting %s' % self.file)
        subprocess.check_call(['tar', 'xf', self.file])

        os.environ['PATH'] = os.environ['PATH'] + ':' + self.prefix + '/bin'
        os.environ['PKG_CONFIG_PATH'] = self.prefix + '/lib/pkgconfig'
        os.environ['CFLAGS'] = '-fPIC'

    def build(self, *params, **kwargs):
        self._log.info('Building %s into %s' % (self.dir, self.prefix))
        if 'sed' in kwargs:
            for k, v in kwargs['sed'].items():
                self._run(['sed', '-i', v, k])
        self._run(['./configure', '--prefix=' + self.prefix] + list(params))
        self._run(['make', '-j', str(CORES), 'install'])

    def rebuild_static_binary(self, file_name):
        self._log.info('Rebuilding %s to be static' % file_name)
        os.unlink(self.dir + '/' + file_name)

        build_command = self._run(['make', file_name, 'V=1'])
        lines = build_command.splitlines()
        line = lines[-1].decode('utf-8')
        line = re.sub(r'^libtool: link: ', '', line)
        line = re.sub(r'\.(so|dylib)\b', '.a', line)
        if platform.system() != 'Darwin':
            line = line + ' -static'
        self._run(line, shell=True)
        self._run(['strip', file_name])
        self._run(['make', 'install'])


def build_opusenc():
    sources = {
        'pkg': Source('https://pkg-config.freedesktop.org/releases/pkg-config-0.29.2.tar.gz'),
        'flac': Source('https://ftp.osuosl.org/pub/xiph/releases/flac/flac-1.3.3.tar.xz'),
        'tools': Source('https://archive.mozilla.org/pub/opus/opus-tools-0.2.tar.gz'),
        'file': Source('https://downloads.xiph.org/releases/opus/opusfile-0.12.tar.gz'),
        'enc': Source('https://archive.mozilla.org/pub/opus/libopusenc-0.2.1.tar.gz'),
        'opus': Source('https://archive.mozilla.org/pub/opus/opus-1.3.1.tar.gz'),
        'ogg': Source('https://downloads.xiph.org/releases/ogg/libogg-1.3.4.tar.gz'),
    }

    for name, source in sources.items():
        source.prepare()

    sources['pkg'].build('--with-internal-glib')
    sources['opus'].build()
    sources['ogg'].build(sed={'include/ogg/os_types.h': '/__APPLE__/a#include <stdint.h>'})
    sources['flac'].build('--enable-static')
    sources['enc'].build()
    sources['file'].build('--disable-http')
    sources['tools'].build('--disable-pie')
    sources['tools'].rebuild_static_binary('opusenc')

    return sources['tools'].prefix + '/bin/opusenc'


if __name__ == '__main__':
    logging.basicConfig(level=logging.DEBUG, format='===== %(levelname)10s %(message)s')
    build_opusenc()
