# coding: utf-8

import logging

import sandbox.sdk2 as sdk2
from sandbox.projects.music.deployment.helpers.AbcHelper import AbcHelper
from sandbox.projects.music.deployment.helpers.ArcadiaHelper import ArcadiaHelper
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper


class MusicCreateStartrekReleaseIssue(MusicBaseTask, TaskHelper):
    class Requirements(sdk2.Task.Requirements):
        environments = [TaskHelper.startrek_client_environment]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60
        description = "Create a Music Startrek Release Issue"

        branch = sdk2.parameters.String("The release branch name", required=True)
        revision = sdk2.parameters.Integer("The release revision", required=True)
        ci_url = sdk2.parameters.String("The CI url", required=True)

        with sdk2.parameters.Output():
            issue = sdk2.parameters.String("Startrek issue")

        abc_token = sdk2.parameters.YavSecret("ABC OAuth token secret", required=True,
                                              description="Yav secret with ABC token",
                                              default='sec-01fp7s63sd8dgjga0erprk7tfx')

        abc_token_key = sdk2.parameters.String("Key for abc token in yav secret", required=True,
                                               default="abc_token")

    def make_new_issue(self, token):
        logging.info('Creating a new issue from %s/%s', self.Parameters.branch, self.Parameters.revision)

        startrek = StartrekHelper(token)
        arcadia = ArcadiaHelper(startrek)

        parsed_changelog, commits_without_task, first_rev, _ = arcadia.prepare_and_extract_latest_changelog(
            self.Parameters.branch)

        abc_token = self.Parameters.abc_token.data()[self.Parameters.abc_token_key]
        self.abc_helper = AbcHelper(abc_token)
        on_duty = self.backend_on_duty()
        text = arcadia.render_changelog(parsed_changelog=parsed_changelog,
                                        commits_without_tasks=commits_without_task,
                                        from_revision=first_rev,
                                        to_revision=self.Parameters.revision,
                                        branch=self.Parameters.branch,
                                        on_duty=on_duty)
        key = startrek.create_issue(CONFIG.issue_summary(self.Parameters.revision, self.Parameters.branch),
                                    text,
                                    CONFIG.checklist,
                                    assignee=on_duty,
                                    qa_engineer=self.qa_on_duty())
        return key

    def on_execute(self):
        token = sdk2.Vault.data(CONFIG.token)
        self.check_authorization(self.author, CONFIG.token, CONFIG.auth_build)

        issue = self.make_new_issue(token)
        self.Parameters.issue = issue.key
        self.Context.issue = issue.key  # for testenv
        self.set_info('Created issue <a href="https://st.yandex-team.ru/{0}">{0}</a>'.format(issue.key),
                      do_escape=False)
