# coding: utf-8

import six

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.sdk2 as sdk2
from sandbox.common import errors
from sandbox.projects.common.vcs import aapi
from sandbox.projects.music.deployment.MusicBuildJars import MusicBuildJars
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.DashboardUpdater import DashboardUpdater
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.projects.music.deployment.helpers.Nyan import nyan, MUSIC_CONTENT_CHAT_ID
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper


class MusicDeployTrunkToTesting(MusicBaseTask, TaskHelper):
    """ Build and deploy Music trunk to one of the test environments """
    fail_on_any_error = True

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60  # real time ~1 min

        description = "Build and deploy Music trunk to one of the test environments"

        url = sdk2.parameters.String("URL to build from",
                                     default="arcadia:/arc/trunk/arcadia",
                                     description=CONFIG.arcadia_description)

        with sdk2.parameters.CheckGroup("Deployment target(s)", required=True) as target:
            for name in (CONFIG.components + CONFIG.components_content):
                setattr(target.values, name, target.Value(name))

    def on_execute(self):
        token = sdk2.Vault.data(CONFIG.token)
        self.check_authorization(self.author, CONFIG.token, CONFIG.auth_build)

        url = self.Parameters.url
        target_dashboards = ['music-' + x for x in self.Parameters.target]

        if not target_dashboards:
            raise errors.TaskFailure('At least one target must be specified')

        with self.memoize_stage.build():
            if url.find('@') == -1:
                revision = aapi.ArcadiaApi.svn_head()
                url = '{}@{}'.format(url, revision)
            self.set_info('Building <b>{}</b>'.format(url), do_escape=False)

            build_task = MusicBuildJars(self,
                                        description='Building {}'.format(url),
                                        url=url,
                                        notifications=self.Parameters.notifications)
            build_task.enqueue()

            self.Context.build_task_id = build_task.id
            raise sdk2.WaitTask(build_task, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK)

        with self.memoize_stage.deploy():
            build_task = sdk2.Task[self.Context.build_task_id]
            if build_task.status not in ctt.Status.Group.SUCCEED:
                raise errors.TaskFailure('Build failed, see the child task #{} for details'.format(build_task.id))

            updater = DashboardUpdater(token)
            for target_dashboard in target_dashboards:
                id_ = updater.set_and_update(self.author,
                                             target_dashboard,
                                             [CONFIG.recipes['testing']],
                                             build_task.id,
                                             url)
                message = u'Started a deployment to <a href="' + \
                          CONFIG.deployment_url(target_dashboard, id_) + \
                          u'">{0}</a>'.format(target_dashboard)
                self.set_info(message, do_escape=False)

            is_content = False
            for name in self.Parameters.target:
                if name in CONFIG.components_content:
                    is_content = True
                    break
            text = u'🛴 Качу тестинг ' + u', '.join(list(self.Parameters.target)) + \
                   u' из транка по просьбе ' + six.text_type(self.author)
            nyan(text)
            if is_content:
                nyan(text, chat_id=MUSIC_CONTENT_CHAT_ID)
            return

        raise errors.TaskFailure('This task is not restartable')
