# coding: utf-8

import StringIO
import logging as log

import sandbox.common.types.misc as ctm
import sandbox.sdk2 as sdk2
from sandbox.common import errors
from sandbox.projects.music.deployment.helpers.AbcHelper import AbcHelper
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.projects.music.deployment.helpers.Nyan import nyan
from sandbox.projects.music.deployment.helpers.StaffHelper import StaffHelper
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper


class MusicDutyNotifier(MusicBaseTask, TaskHelper):
    """Dummy task for testing purposes"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        description = "Send music on duty to telegram"

        duty_ended_notification = sdk2.parameters.Bool("Send message that duty has ended",
                                                       default=False)  # type: bool
        old_style_form = sdk2.parameters.Bool("Use one abc service for duty", default=True)
        with duty_ended_notification.value[False]:
            tokens = sdk2.parameters.YavSecret("ABC and Staff OAuth token", required=True,
                                               description="Yav secret with 'abc_token' and 'staff_oauth' keys")
            message_header = sdk2.parameters.String("Message header", default='Дежурные:', required=True)

            with old_style_form.value[True]:
                abc_service = sdk2.parameters.String("ABC service slug", required=True,
                                                     description="ABC sevices slug for the duty scheduler. Ex: musicbackend")  # type: str
                abc_schedules = sdk2.parameters.Dict("ABC schedule slugs", required=True,
                                                     description="Schedule slug from ABC group and custom tile. "
                                                                 "Ex: music-infra, 'Инфра'")
            with old_style_form.value[False]:
                abc_services = sdk2.parameters.Dict("ABC service slug + schedule slugs", required=True,
                                                    description="format -> service_slug:schedules_slug, message")

        telegrams_ids = sdk2.parameters.List('Telegram ids', required=True,
                                             description='Telegram ids to send notification to',
                                             value_type=sdk2.parameters.Integer)
        silent = sdk2.parameters.Bool("Send quietly", default=True)  # type: bool

    @staticmethod
    def _build_message(on_duty, message):
        output = StringIO.StringIO()

        output.write(message + "\n")

        for d in on_duty:
            output.write(d + "\n")

        contents = output.getvalue()
        output.close()

        return contents

    @staticmethod
    def _message_template(title, people_on_duty, staff_helper):
        if len(people_on_duty) > 0 and people_on_duty[0]['person']:
            person = people_on_duty[0]['person']
            telegram = staff_helper.get_user_telegram(person['login'])
            template = "{title} -> [{name}](https://staff.yandex-team.ru/{login}) ([{telegram}](telegram.me/{" \
                       "telegram}))"
            return template.format(title=title, name=person["name"]["ru"], login=person['login'], telegram=telegram)
        else:
            template = "{title} -> Дежурный не назначен"
            return template.format(title=title)

    def duty_ended_message(self):
        message = "=========================================\n" \
                  "Работа с обращениями на сегодня завершена\n" \
                  "\"/music\" - призвать дежурных при инциденте.\n" \
                  "========================================="
        try:
            for t_id in self.Parameters.telegrams_ids:
                nyan(message, chat_id=t_id, disable_notification=self.Parameters.silent, raise_exception=True)
        except Exception as ex:
            raise errors.TaskError(ex)

    def on_duty_message(self):
        token = self.Parameters.tokens.data()
        abc_helper = AbcHelper(token["abc_token"])
        staff_helper = StaffHelper(token["staff_oauth"])

        on_duty = []
        if self.Parameters.old_style_form:
            for slug, title in self.Parameters.abc_schedules.items():
                on_duty.append(self._message_template(title, abc_helper.get_onduty(self.Parameters.abc_service, slug),
                                                      staff_helper))
        else:
            for slugs, title in self.Parameters.abc_services.items():
                abc_slug = slugs.split(":")[0]
                abc_schedule_slug = slugs.split(":")[1]
                on_duty.append(self._message_template(title, abc_helper.get_onduty(abc_slug, abc_schedule_slug),
                                                      staff_helper))

        out = self._build_message(on_duty, self.Parameters.message_header)
        log.info(out)
        try:
            for t_id in self.Parameters.telegrams_ids:
                nyan(out, chat_id=t_id, disable_notification=self.Parameters.silent, raise_exception=True)
        except Exception as ex:
            raise errors.TaskError(ex)

    def on_execute(self):
        if self.Parameters.duty_ended_notification:
            self.duty_ended_message()
        else:
            self.on_duty_message()
