# coding: utf-8
import re
from textwrap import dedent

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.task import Status
from sandbox.projects.common import link_builder as lb
from sandbox.projects.music.deployment.helpers.AbcHelper import AbcHelper
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper


class MusicExportTest(sdk2.Task):
    _issue_re = re.compile(r'^MUSICBACKEND-\d+$')

    class Requirements(sdk2.Task.Requirements):
        environments = [TaskHelper.startrek_client_environment]

    class Parameters(sdk2.Task.Parameters):
        description = "Run export service tests"

        arcadia_url = sdk2.parameters.String("The arcadia url",
                                             default="arcadia:/arc/trunk/arcadia",
                                             description=CONFIG.arcadia_description,
                                             required=True)

        arcadia_patch = sdk2.parameters.String("The patch file to apply", required=False)
        issue = sdk2.parameters.String("Startrek issue to inform to", required=False)  # type: str

        export_url = sdk2.parameters.String("Export page base url",
                                            default="https://base.music.qa.yandex-team.ru",
                                            description="default https://base.music.qa.yandex-team.ru, "
                                                        "e.g. https://export-pr-x.music.qa.yandex-team.ru",
                                            required=True)  # type: str

        abc_token = sdk2.parameters.YavSecret("ABC OAuth token secret", required=True,
                                              description="Yav secret with ABC tolen",
                                              default='sec-01fp7s63sd8dgjga0erprk7tfx')

        abc_token_key = sdk2.parameters.String("Key for abc token in yav secret", required=True,
                                               default="abc_token")

        with sdk2.parameters.Output:
            allure_report = sdk2.parameters.String("Allure report url")

    def create_subtask(self):
        task_class = sdk2.Task["YA_MAKE_2"]

        arcadia_patch = "zipatch:" + str(self.Parameters.arcadia_patch) if self.Parameters.arcadia_patch else None

        sub_task = task_class(self,
                              result_rd="Test music admin",
                              checkout="True",
                              targets="music/backend/music-common/test-api-export",
                              test="True",
                              arcadia_patch=arcadia_patch,
                              test_log_level="debug",
                              test_failure_code="10",
                              tests_retries="1",
                              build_system="ya",
                              test_tag="ya:manual",
                              allure_report="True",
                              allure_report_ttl=21,
                              checkout_arcadia_from_url=str(self.Parameters.arcadia_url),
                              force_build_depends="True",
                              definition_flags="-DJDK_VERSION={}".format(CONFIG._DEFAULT_JDK_VERSION),
                              env_vars="ROBOT_MUSIC_BACKEND_EXPORT_TOKEN='$(vault:value:MUSIC:ROBOT_MUSIC_BACKEND_EXPORT_TOKEN)' " +
                                       "EXPORT_URL=" + self.Parameters.export_url,
                              keep_on="True")
        sub_task.enqueue()
        self.Context.subtask = sub_task.id
        self.Context.save()
        raise sdk2.WaitTask(sub_task.id, Status.Group.FINISH + Status.Group.BREAK)

    def check_subtask(self):
        token = sdk2.Vault.data(CONFIG.token)
        st = StartrekHelper(token)

        sub_task = sdk2.Task.find(id=self.Context.subtask).first()
        allure_report = sdk2.Resource["ALLURE_REPORT"].find(task=sub_task).first().http_proxy + "/index.html"

        msg = dedent(u'Subtask {subtask} status: {task_status}\n'
                     u'Allure report: <a href="{allure_report}">index.html</a>'
                     .format(subtask=lb.task_link(sub_task.id, plain=False),
                             task_status=sub_task.status,
                             allure_report=allure_report))

        self.set_info(info=msg, do_escape=False)

        self.Parameters.allure_report = allure_report

        if sub_task.status == Status.SUCCESS:
            if self.Parameters.issue:
                st.add_comment(self.Parameters.issue,
                               u'Тест admin прошел успешно\n'
                               u'Sandbox task: ((https://sandbox.yandex-team.ru/task/{task_id} {task_id}))\n'
                               u'Allure report: (({allure_report} index.html))'
                               .format(task_id=self.id,
                                       allure_report=allure_report))
                st.patch_release_issue_statuses_raw(self.Parameters.issue, CONFIG.monitored_deployment_test_names,
                                                    self.__class__.__name__)
            return
        else:
            if self.Parameters.issue:

                th = TaskHelper()
                abc_token = self.Parameters.abc_token.data()[self.Parameters.abc_token_key]
                th.abc_helper = AbcHelper(abc_token)

                if self._issue_re.match(self.Parameters.issue):
                    summonee = st.ticket_assignee(self.Parameters.issue)
                else:
                    summonee = th.backend_on_duty()

                st.add_comment(self.Parameters.issue,
                               u'Тест admin !!сломался!!\n'
                               u'Sandbox: ((https://sandbox.yandex-team.ru/task/{task_id} {task_id}))\n'
                               u'Allure report: (({allure_report} index.html))'
                               .format(task_id=self.id,
                                       allure_report=allure_report),
                               summonee=summonee)
            raise errors.TaskFailure()

    def on_execute(self):
        with self.memoize_stage.create_subtask:
            # will run only once
            self.create_subtask()

        self.check_subtask()
