# coding: utf-8
import logging
import time
import uuid

import requests

import sandbox.sdk2 as sdk2
from sandbox.common import errors
from sandbox.projects.music.deployment.helpers.AbcHelper import AbcHelper
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper
from sandbox.projects.music.deployment.helpers.TestConfig import TEST_CONFIG


class RequestHelper(object):
    def __init__(self, token):
        self.token = token

    def get(self, url):
        rid_header = 'X-Request-Id'
        headers = {
            'Authorization': 'OAuth ' + self.token,
            'User-Agent': 'MusicExportTest',
            rid_header: str(uuid.uuid4())
        }
        try:
            res = requests.get(url, headers=headers, verify=False)
            res.raise_for_status()
        except:
            logging.exception("Something went wrong {}, rid: {}".format(url, headers[rid_header]), exc_info=True)
            raise
        else:
            return res


# Deprecated
class MusicExportTestOld(sdk2.Task, TaskHelper):
    """Music export service test"""

    class Requirements(sdk2.Task.Requirements):
        environments = [TaskHelper.startrek_client_environment]
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30 * 60  # seconds
        description = "Test export service"

        issue = sdk2.parameters.String("Startrek issue to inform to", required=True)

        abc_token = sdk2.parameters.YavSecret("ABC OAuth token secret", required=True,
                                              description="Yav secret with ABC token",
                                              default='sec-01fp7s63sd8dgjga0erprk7tfx')

        abc_token_key = sdk2.parameters.String("Key for abc token in yav secret", required=True,
                                               default="abc_token")

        debug_mode = sdk2.parameters.Bool("Debug mode. Do not summon on duty developer", default=False)

    @staticmethod
    def assert_response_ok(request_helper, url):
        # type: (RequestHelper, str) -> None

        logging.info("Executing " + url)
        response = request_helper.get(url)
        raw_response = response.text
        if "Struts Problem Report" in raw_response:
            raise errors.TaskFailure('Struts error detected on {}'.format(url))
        if "stack trace" in raw_response or "StackTrace" in raw_response:
            raise errors.TaskFailure('Stack trace detected on {}'.format(url))
        if "FreeMarker template error" in raw_response:
            raise errors.TaskFailure('FTL error detected on {}'.format(url))

    def on_execute(self):
        st_token = sdk2.Vault.data(CONFIG.token)
        st = StartrekHelper(st_token)

        export_token = sdk2.Vault.data(CONFIG.export_token)
        req = RequestHelper(export_token)

        try:
            for host in TEST_CONFIG.export_qa_hosts:
                for url in TEST_CONFIG.export_qa_urls:
                    time.sleep(0.5)
                    MusicExportTestOld.assert_response_ok(req, host + url)

            for host in TEST_CONFIG.origs_test_endpoints:
                for url in TEST_CONFIG.origs_test_urls:
                    MusicExportTestOld.assert_response_ok(req, host + url)

            st.add_comment(self.Parameters.issue,
                           u'Тест на админку прошел успешно: ((https://sandbox.yandex-team.ru/task/{0} {0}))'
                           .format(self.id))
            st.patch_release_issue_statuses_raw(self.Parameters.issue, CONFIG.monitored_deployment_test_names,
                                                self.__class__.__name__)
        except BaseException:
            abc_token = self.Parameters.abc_token.data()[self.Parameters.abc_token_key]
            self.abc_helper = AbcHelper(abc_token)

            no_duty = self.backend_on_duty() if not self.Parameters.debug_mode else None

            st.add_comment(self.Parameters.issue,
                           u'Тест на админку !!сломался!!: ((https://sandbox.yandex-team.ru/task/{0} {0}))'
                           .format(self.id),
                           summonee=no_duty)
            raise
