# coding: utf-8
import logging
import time

import requests
from requests.auth import HTTPBasicAuth

import sandbox.sdk2 as sdk2
from sandbox.common import errors
from sandbox.projects.music.deployment.helpers.AbcHelper import AbcHelper
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper


class ZAdmin(object):
    def __init__(self, token, groups, port):
        self.token = token
        self.host = self.alive_host(groups)
        self.port = port
        logging.info('Found an alive host: {}'.format(self.host))

    def raw_invoke(self, host, port, bean, method, param=''):
        url = "https://{host}:{port}/z/manager/bean/{bean}/method/{method}/invoke.json?log=false&p0={param}"
        full_url = url.format(host=host,
                              port=port,
                              bean=bean,
                              method=method,
                              param=param)
        logging.debug('Requesting: {}'.format(full_url))
        response = self.request_host(full_url, self.token)
        return response.json()['invocationInfo']['result']

    def raw_invoke_blobs_info(self, host, port):
        url = "https://{host}:{port}/blobs/info"
        full_url = url.format(host=host, port=port)
        logging.debug('Requesting: {}'.format(full_url))
        response = self.request_host(full_url, self.token)
        json = response.json()
        deployed = len(json['disk']['entries'])
        cached = len(json['memory']['entries'])
        logging.info('Blobs status: {} deployed, {} cached'.format(deployed, cached))
        return json

    @staticmethod
    def request_host(url, token):
        r = requests.get(url,
                         auth=HTTPBasicAuth('music-admin', 'music-admin'),
                         headers={
                             'Authorization': 'OAuth ' + token,
                             'Cookie': 'Session_id=123',
                         },
                         allow_redirects=False)
        if not 199 < r.status_code < 299:
            if r.status_code == 302:
                raise requests.HTTPError("Redirected to {}. token={}".format(r.headers["Location"], token[0:10]), response=r)
            raise requests.HTTPError("Not OK: token={}, res={}".format(token[0:10], r), response=r)
        return r

    def invoke(self, bean, method, param=''):
        return self.raw_invoke(self.host, self.port, bean, method, param)

    @staticmethod
    def resolve_nanny(groups):
        hosts = []
        for group in groups:
            r = requests.get('https://nanny.yandex-team.ru/v2/services/{}/current_state/instances/'.format(group))
            r.raise_for_status()
            hosts += [h['container_hostname'] for h in r.json()['result']]
        return hosts

    def alive_host(self, groups):
        hosts = self.resolve_nanny(groups)
        for host in hosts:
            hostname = host.split('.')[0] + '.music.yandex-team.ru'
            try:
                self.raw_invoke(hostname, 81, 'properties', 'size')
                return hostname
            except requests.RequestException as error:
                logging.info('{} returned {}'.format(hostname, error))
        raise errors.TaskFailure('Cannot find an alive host in {}'.format(groups))


class App(ZAdmin):
    def __init__(self, token):
        super(App, self).__init__(token, ['music_test_back_' + dc for dc in ['man', 'sas', 'vla']], 81)
        self.blob_registry_bean = 'faceBlobRegistry'

    def is_deployed_on_filesystem(self, blob_name):
        result = self.raw_invoke_blobs_info(self.host, self.port)
        for deployed_blob_info in result['disk']['entries']:
            deployed_blob_name = deployed_blob_info['key']
            if deployed_blob_name == blob_name:
                return True
        return False

    def blob_registry_is_cached(self, blob_name):
        result = self.raw_invoke_blobs_info(self.host, self.port)
        cached = len(result['memory']['entries'])
        if cached == 1:
            cached_blob_name = result['memory']['entries'][0]['key']
            if cached_blob_name == blob_name:
                return True
        return False

    def await_local_deployment(self, blob_name, cycles, period):
        logging.info("Awaiting deployment of statics with prefix " + blob_name)
        for i in range(0, cycles):
            logging.info("Checking local deployment of " + blob_name + " on " + self.host)
            if self.is_deployed_on_filesystem(blob_name):
                logging.info("Done")
                return
            time.sleep(period)
        raise errors.TaskFailure("No local deployment within specified time!")

    def await_preload(self, blob_name, cycles, period):
        logging.info("Awaiting preload of statics with prefix " + blob_name)
        for i in range(0, cycles):
            logging.info("Checking if blob " + blob_name + " is cached on " + self.host)
            if self.blob_registry_is_cached(blob_name):
                logging.info("Done")
                return
            time.sleep(period)
        raise errors.TaskFailure("No preload within specified time!")


class Worker(ZAdmin):
    def __init__(self, token):
        super(Worker, self).__init__(token, ['music_test_heavy_worker_' + dc for dc in ['man', 'sas', 'vla']], 81)

    def current_prefix(self, prefix_type):
        return self.invoke("facegenPrefixManager", "currentPrefix-bcb08aa4", prefix_type)

    def staticsgen_admin_schedule(self, localtime):
        logging.info("Scheduling generation of statics for " + localtime)
        return self.invoke("staticsgenAdmin", "schedule", localtime)


class MusicFacedeployTest(sdk2.Task, TaskHelper):
    """The Music statics deployment test"""

    class Requirements(sdk2.Task.Requirements):
        environments = [TaskHelper.startrek_client_environment]
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 200 * 60
        description = "Test statics deployment"

        issue = sdk2.parameters.String("Startrek issue to inform to", required=True)

        abc_token = sdk2.parameters.YavSecret("ABC OAuth token secret", required=True,
                                              description="Yav secret with ABC token",
                                              default='sec-01fp7s63sd8dgjga0erprk7tfx')

        abc_token_key = sdk2.parameters.String("Key for abc token in yav secret", required=True,
                                               default="abc_token")

    def on_execute(self):
        token = sdk2.Vault.data(CONFIG.token)
        st = StartrekHelper(token)

        try:
            http_token = sdk2.Vault.data(CONFIG.export_token)
            localtime = time.strftime('%Y-%m-%dT%H:%M:%S')
            blob_name = 'facegen-' + localtime.replace(':', '-')

            worker = Worker(http_token)
            app = App(http_token)

            self.set_info('Generating a schedule')
            worker.staticsgen_admin_schedule(localtime)

            self.set_info('Waiting for deployment')
            app.await_local_deployment(blob_name, 240, 120)
            self.set_info('Waiting for preload')
            app.await_preload(blob_name, 180, 180)
            self.set_info('Success!')

            st.add_comment(self.Parameters.issue,
                           u'Тест раскладки статики прошел успешно: ((https://sandbox.yandex-team.ru/task/{0} {0}))'
                           .format(self.id))
            st.patch_release_issue_statuses_raw(self.Parameters.issue, CONFIG.monitored_deployment_test_names,
                                                self.__class__.__name__)
        except BaseException:
            abc_token = self.Parameters.abc_token.data()[self.Parameters.abc_token_key]
            self.abc_helper = AbcHelper(abc_token)
            st.add_comment(self.Parameters.issue,
                           u'Тест раскладки статики !!сломался!!: ((https://sandbox.yandex-team.ru/task/{0} {0}))'
                           .format(self.id),
                           summonee=self.backend_on_duty())
            raise
