# coding: utf8
import base64
import logging
import os
import shutil
import threading
from typing import Dict

import sandbox.common.types.misc as ctm
import sandbox.projects.music.resources as mres
import sandbox.common.types.client as ctc
import sandbox.projects.common.environments as cte

import sandbox.sdk2 as sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.vcs.git import Git


class NodeGeobaseBindingsEnvironment(sdk2.environments.TarballToolkitBase):
    name = "NodeGeobaseBindings"
    resource_type = "MUSIC_NODE_GEOBASE_BUILD_RESOURCE"

    def check_resource_platform(self, resource, allow_binary_compatible=False):
        resource_platform = getattr(resource, "platform", "")
        if resource_platform == "":
            return True
        return super(NodeGeobaseBindingsEnvironment, self).check_resource_platform(resource, allow_binary_compatible)

    def prepare(self):
        basedir = super(NodeGeobaseBindingsEnvironment, self).prepare()
        self.update_os_env('CPATH', os.path.abspath(os.path.join(basedir, "include")))
        self.update_os_env('LIBRARY_PATH', os.path.abspath(os.path.join(basedir, "lib")))


class NoPlatformCheckNodeJsEnvironment(sdk2.environments.NodeJS):
    def check_resource_platform(self, resource, allow_binary_compatible=False):
        resource_platform = getattr(resource, "platform", "")
        if resource_platform == "":
            return True
        return super(NoPlatformCheckNodeJsEnvironment, self).check_resource_platform(resource, allow_binary_compatible)


class MusicNodeGeobaseBuildResource(mres.MusicDefaultResource):
    pass


class MusicFrontRadioResource(mres.MusicDefaultResource):
    pass


class MusicFrontMtsResource(mres.MusicDefaultResource):
    pass


class MusicFrontSandboxResource(mres.MusicDefaultResource):
    pass


targets = {
    'radio': MusicFrontRadioResource,
    'mts': MusicFrontMtsResource,
    'sandbox': MusicFrontSandboxResource,
}  # type: Dict[str, mres.MusicDefaultResource.__class__]


class MusicFrontBuild(sdk2.Task):
    git_dir = 'frontend-mono'

    class Requirements(sdk2.Requirements):
        environments = (
            NoPlatformCheckNodeJsEnvironment('14.12.0'),
            NodeGeobaseBindingsEnvironment('6.0-34'),
            cte.SandboxGitEnvironment('2.19.0'),
        )
        disk_space = 1024 * 5
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_XENIAL

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 20 * 60
        description = "Build Yandex.Music Frontend"

        with sdk2.parameters.Group("Git parameters") as git_block:
            repository = sdk2.parameters.String("Repository", required=True,
                                                default="ssh://git@bb.yandex-team.ru/music/frontend-mono.git")
            ref_id = sdk2.parameters.String("Ref id", default="trunk")

        with sdk2.parameters.Group("Vault") as vault_block:
            ssh_secret = sdk2.parameters.YavSecret("Git access ssh key", default="sec-01cyh6gvp0t1nfbx0682qvwvbb",
                                                   description="Yav secret with the ssh key to check out")
            ssh_key = sdk2.parameters.String("The key name", default="id_rsa_passwordless")

    def _checkout(self):
        self.set_info('Checking out repo')
        secret64 = self.Parameters.ssh_secret.data()[self.Parameters.ssh_key]
        secret = base64.b64decode(secret64)
        with sdk2.ssh.Key(self, private_part=secret):
            git = Git(self.Parameters.repository)
            git.clone(self.git_dir, self.Parameters.ref_id)

    def _global_prepare(self):
        self.set_info('Installing dependencies')
        self._run(['git', 'clean', '-df'], self.git_dir)
        self._run(['npm', '-g', 'install', 'yarn'], self.git_dir)
        self._run(['yarn', 'install'], self.git_dir)
        os.mkdir('out')

    def _run(self, args, cwd=None):
        """
        @type args: list
        @type cwd: str
        """
        name = '{}:: {}'.format(os.path.basename(cwd) if cwd else '-', ' '.join(args))
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(name)) as pl:
            subprocess.check_call(args, cwd=cwd, stdout=pl.stdout, stderr=subprocess.STDOUT)

    def _build_target(self, target):
        """
        @type target: str
        """
        full_path = os.path.join(self.git_dir, 'lib/fronts', target)
        try:
            # Try to build it
            if os.path.exists(os.path.join(full_path, '.prepare.js')):
                self.set_info('Preparing {} for build'.format(target))
                self._run(['node', '.prepare.js'], full_path)
            if os.path.exists(os.path.join(full_path, '.build.js')):
                self.set_info('Building {}'.format(target))
                self._run(['node', '.build.js'], full_path)

            # At this point we have a built project in .build/dist and several symlinks
            # are broken there. We move it to the correct directory to fix the links, add
            # deps and package it out.
            tmp_path = os.path.join(os.path.dirname(full_path), target + '__buildtmp')
            try:
                os.rename(os.path.join(full_path, '.build', 'dist'), tmp_path)
                shutil.rmtree(os.path.join(full_path, '.build'))

                self.set_info('Installing {} dependencies'.format(target))
                # yarn keeps local cache that must not be simultaneously updated
                with self.yarn_mutex:
                    self._run(['yarn', 'install'], tmp_path)

                # we need to transfer the result into task's cwd to package the resource
                out_path = os.path.abspath(os.path.join('out', target + '.tar.gz'))
                self._run(['tar', 'cfz', out_path, '-C', tmp_path, '.'])
            finally:
                shutil.rmtree(tmp_path)
            resource = targets[target](self,
                                       description='Resource for {}'.format(target),
                                       path=out_path,
                                       arch=None,
                                       ttl=14)
            data = sdk2.ResourceData(resource)
            data.ready()

            self.set_info(u'✅ Target {} has been built successfully'.format(target))
        except BaseException as x:
            self.set_info(u'⛔️ Target {} has failed: {}'.format(target, x))
            self.error = x

    def on_execute(self):
        self._checkout()
        self._global_prepare()

        self.error = None
        self.yarn_mutex = threading.Lock()
        tasks = [threading.Thread(target=self._build_target, args=(x,)) for x in targets]
        for task in tasks:
            task.start()
        for task in tasks:
            task.join()
        if self.error:
            raise self.error
