# coding: utf-8

import sandbox.sdk2 as sdk2
from sandbox.projects.music.MusicBackupYdb import MusicBackupYdb
from sandbox.projects.music.MusicRestoreYdb import MusicRestoreYdb
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper, subtaskable
from sandbox.projects.music.deployment.helpers.YdbHelper import YdbHelper


class MusicMigrateYdbProdToQa(MusicBaseTask, TaskHelper):
    """Migrates all tables with their data from one database to another"""

    ALLOWED_TABLE_OWNERS = {'robot-muz-ydb-qa@staff', 'robot-music-ydb@staff', 'robot-ydb-cp@staff', 'noobgam@staff', 'robot-music-ynison@staff', 'robot-muz-ynison-qa@staff'}

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60
        description = "Migrate tables from prod to qa"

        ydb_prod_database_id = sdk2.parameters.String(
            "YDB prod database name",
            default_value="etn024qit0i7rmv5kgco",
            required=True
        )

        ydb_qa_database_id = sdk2.parameters.String(
            "YDB qa database id",
            default_value="etn024qit0i7rmv5kgco",
            required=True,
        )

        ydb_qa_database_name = sdk2.parameters.String(
            "YDB qa database name",
            default_value="/ru-prestable/musicbackend/qa/music/",
            required=True,
        )

        ydb_qa_endpoint = sdk2.parameters.String(
            "YDB qa endpoint",
            default_value="ydb-ru-prestable.yandex.net:2135",
            required=True,
        )

        cloud_name = sdk2.parameters.String(
            "cloud name",
            description="Cloud name used in YC containing both databases handled by migration",
            default_value="musicbackend"
        )

        ydb_token_name = sdk2.parameters.String(
            "ydb token name",
            default_value='ydb_prod_token',
            description="Token name to extract from YAV, "
                        "necessary if you want to import to an environment other than QA"
        )

        token = sdk2.parameters.YavSecret(
            "YAV secret with ydb-yt migrations and backups",
            default='sec-01ej69jtmv3t8g2675y81dc5bv',
            required=True
        )

        yc_binary = sdk2.parameters.Resource(
            "yc binary"
        )

    @subtaskable(True)
    def backup_prod(self):
        self.enqueue_subtask(
            MusicBackupYdb,
            description="Regular prod -> qa migration. QA restore",
            cloud_name=self.Parameters.cloud_name,
            yc_binary=self.Parameters.yc_binary,
            token=self.Parameters.token,
            database=self.Parameters.ydb_prod_database_id,
        )

    @subtaskable(True)
    def restore_backup(self, music_backup_ydb):
        with self.memoize_stage.drop_tables:
            ydb_token = self.Parameters.token.data()[self.Parameters.ydb_token_name]

            yh = YdbHelper(
                ydb_token,
                self.Parameters.ydb_qa_endpoint,
                self.Parameters.ydb_qa_database_name,
                None,  # yt token is not needed there
                None,
                self
            )
            tables_list = MusicMigrateYdbProdToQa.__extract_tables_from_ls(yh.ls())
            for table in tables_list:
                yh.drop_table(table)

        backup_id = music_backup_ydb.Context.backup_id
        self.enqueue_subtask(
            MusicRestoreYdb,
            description="Regular prod -> qa migration. QA restore",
            cloud_name=self.Parameters.cloud_name,
            yc_binary=self.Parameters.yc_binary,
            token=self.Parameters.token,
            database=self.Parameters.ydb_qa_database_id,
            backup_id=backup_id
        )

    @subtaskable(True)
    def wait_for_tasks_to_complete(self, music_restore_ydb):
        pass

    def on_execute(self):
        self.backup_prod()
        self.restore_backup()
        self.wait_for_tasks_to_complete()

    @staticmethod
    def __extract_tables_from_ls(data):
        # no json format available before https://st.yandex-team.ru/KIKIMR-10435
        lines = filter(lambda line: not not line, data.splitlines())
        # extract column names from table output
        column_number = {}
        import re
        idx = 0
        for col in lines[1].split('|')[1:-1]:
            m = re.search('\w+', col)
            column_number[m.group(0)] = idx
            idx += 1
        tables = []
        for row in lines[3:-1]:
            columns = row.split('|')[1:-1]
            owner = columns[column_number['Owner']].strip()
            path = columns[column_number['Name']].strip()
            col_type = columns[column_number['Type']].strip()
            if col_type != 'table' or owner not in MusicMigrateYdbProdToQa.ALLOWED_TABLE_OWNERS:
                continue
            tables += [path]
        return tables
