# coding: utf-8
import re
import logging
from textwrap import dedent

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.task import Status
from sandbox.projects.common import link_builder as lb
from sandbox.projects.music.deployment.helpers.AbcHelper import AbcHelper
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper


class MusicQuasarSampleAppTest(MusicBaseTask):
    _issue_re = re.compile(r'^(MUSICBACKEND|MEDIASEARCH)-\d+$')

    class Requirements(sdk2.Task.Requirements):
        environments = [TaskHelper.startrek_client_environment]

    class Parameters(sdk2.Task.Parameters):
        description = "Run quasar_sample_app tests"

        issue = sdk2.parameters.String("Startrek issue to inform to", required=False)  # type: str

        arcadia_patch = sdk2.parameters.String("The patch file to apply", required=False)
        backend_url = sdk2.parameters.String("music-web external api url",
                                             required=True,
                                             default='http://music-web-ext.music.qa.yandex.net:80')  # type: str

        abc_token = sdk2.parameters.YavSecret("ABC OAuth token secret", required=True,
                                              description="Yav secret with ABC token",
                                              default='sec-01fp7s63sd8dgjga0erprk7tfx')

        abc_token_key = sdk2.parameters.String("Key for abc token in yav secret", required=True,
                                               default="abc_token")

    def create_subtask(self):
        logging.info("Create YA_MAKE subtask")
        task_class = sdk2.Task["YA_MAKE"]
        arcadia_patch = "zipatch:" + str(self.Parameters.arcadia_patch) if self.Parameters.arcadia_patch else None
        api_url = self.Parameters.backend_url
        sub_task = task_class(
            self,
            description="Functional test for music with QA backend",
            arcadia_patch=arcadia_patch,
            checkout_arcadia_from_url="arcadia:/arc/trunk/arcadia",
            use_aapi_fuse=True,
            use_arc_instead_of_aapi=True,
            aapi_fallback=True,
            build_system="semi_distbuild",
            checkout_mode="auto",
            env_vars="TUS_TOKEN=$(vault:value:MUSIC:robot-music-backend-tus-token)",
            test="True",
            test_threads=1,
            test_params=(
                'vins-url="http://vins.hamster.alice.yandex.net/speechkit/app/pa/?'
                'srcrwr=BASS_Music:{api_url}/external-api&'.format(api_url=api_url) +
                'srcrwr=BASS_MusicCatalog:{api_url}/external-api&'.format(api_url=api_url) +
                'srcrwr=BASS_MusicCatalogBulk:{api_url}/external-api&'.format(api_url=api_url) +
                'srcrwr=BASS_MusicQuasar:{api_url}/external-api&'.format(api_url=api_url) +
                'srcrwr=BASS_MusicSuggests:{api_url}/external-api&'.format(api_url=api_url) +
                'srcrwr=BASS_SRCRWR_Music:{api_url}/external-api&'.format(api_url=api_url) +
                'srcrwr=BASS_SRCRWR_Music:{api_url}/external-api&'.format(api_url=api_url) +
                'srcrwr=BASS_SRCRWR_MusicCatalogBulk:{api_url}/external-api&'.format(api_url=api_url) +
                'srcrwr=BASS_SRCRWR_MusicQuasar:{api_url}/external-api&'.format(api_url=api_url) +
                'srcrwr=BASS_SRCRWR_MusicSuggests:{api_url}/external-api&'.format(api_url=api_url) +
                'srcrwr=HOLLYWOOD_MUSIC_BACKEND_PROXY:{api_url}" '.format(api_url=api_url) +
                'music-api-url="wss://ws-api.music.qa.yandex.net/quasar/websocket" '
                'reset_account_music_info=1'
            ),
            test_filters=(
                'test_music_play_pause.py::test_music_play_pause[play_artist.wav] '
                'test_music_play_pause.py::test_music_play_pause[play_genre.wav] '
                'test_music_play_pause.py::test_music_play_pause[play_song.wav] '
                'test_music_play_pause.py::test_music_play_pause[play_podcast.wav] '
                'test_music_play_pause.py::test_music_play_pause[play_morning_show.wav] '
                'test_music_play_pause.py::test_music_play_pause[play_fairytail.wav] '
                'test_music_play_pause.py::test_music_play_pause[play_waterfall_sound.wav] '
                'test_music_play_pause.py::test_music_no_plus test_music_play_pause.py::test_what_playing '
                'test_music_play_pause.py::test_shazam[args0] test_music_play_pause.py::test_shazam[args1] '
                'test_music_like_dislike.py '
                'test_music_next_prev.py'
            ),
            test_log_level="debug",
            ram_drive_size=1024,
            ya_yt_token_vault_owner="MUSIC",
            ya_yt_token_vault_name="ROBOT_MUSIC_BACKEND_YT_TOKEN",
            targets="yandex_io/functional_tests/music",
        )
        sub_task.enqueue()
        self.Context.subtask = sub_task.id
        self.Context.save()
        raise sdk2.WaitTask(sub_task.id, Status.Group.FINISH + Status.Group.BREAK)

    def check_subtask(self):
        token = sdk2.Vault.data(CONFIG.token)
        st = StartrekHelper(token)

        logging.info("Checking subtask status")
        sub_tasks = sdk2.Task.find(id=self.Context.subtask).limit(1)
        sub_task = [task for task in sub_tasks][0]

        if sub_task.status == Status.SUCCESS:
            if self.Parameters.issue:
                st.add_comment(
                    self.Parameters.issue,
                    u'Тест quasar прошел успешно: {}'.format(
                        lb.task_wiki_link(self.id))
                )
                st.patch_release_issue_statuses_raw(self.Parameters.issue, CONFIG.monitored_deployment_test_names,
                                                    self.__class__.__name__)
            return
        else:
            resources = sdk2.Resource["BUILD_OUTPUT_HTML"].find(task=sub_task).limit(10)
            error_log = [res for res in resources][0].http_proxy

            msg = dedent("""
                         Subtask {subtask} failed
                         Error log: <a href="{error_log}">output_1_1.html</a>
                         """.format(subtask=lb.task_link(sub_task.id, plain=False),
                                    error_log=error_log))

            self.set_info(info=msg, do_escape=False)

            if self.Parameters.issue:
                abc_token = self.Parameters.abc_token.data()[self.Parameters.abc_token_key]
                th = TaskHelper()
                th.abc_helper = AbcHelper(abc_token)
                if self._issue_re.match(self.Parameters.issue):
                    summonee = st.ticket_assignee(self.Parameters.issue)
                else:
                    summonee = th.backend_on_duty()

                st.add_comment(self.Parameters.issue,
                               u'Тест quasar !!сломался!!: '
                               u'((https://sandbox.yandex-team.ru/task/{0} {0}))'
                               .format(self.id),
                               summonee=summonee)
            raise errors.TaskFailure()

    def on_execute(self):
        with self.memoize_stage.create_subtask:
            # will run only once
            self.create_subtask()

        self.check_subtask()
