import json
from collections import namedtuple

import requests
from requests.adapters import HTTPAdapter

Zone = namedtuple('Zone', ['zone', 'uuid', 'type'])
Record = namedtuple('Record', ['type', 'ttl', 'left', 'right'])


class DNSApi:
    _base_url = 'https://dns-api.yandex.net/'

    def __init__(self, username, token):
        # type: (str, str) -> DNSApi
        self._username = username
        self._token = token
        self._session = self._make_session()

    @staticmethod
    def _make_session():
        session = requests.Session()
        adapter = HTTPAdapter(max_retries=3)
        session.mount('https://', adapter)
        return session

    def _fetch(self, url, params=None, api_ver='v2.0'):
        full_url = self._base_url + '{}/{}/'.format(api_ver, self._username) + url
        resp = self._session.get(full_url,
                                 params=params,
                                 headers={
                                     'X-Auth-Token': self._token,
                                     'Accept': 'application/json',
                                 })
        resp.raise_for_status()
        return resp.json()

    def _put(self, payloads):
        full_url = self._base_url + 'v2.3/{}/primitives'.format(self._username)

        # no retries on update ops
        resp = requests.put(full_url,
                            data=json.dumps({
                                'primitives': payloads,
                            }),
                            headers={
                                'X-Auth-Token': self._token,
                                'Accept': 'application/json',
                                'Content-type': 'application/json',
                            }, verify=False)

        resp.raise_for_status()
        return resp.json()

    def find_zone(self, hostname):
        # type: (str) -> Zone
        data = self._fetch('zones/map?{}'.format(hostname))
        item = data['zones'][0]
        if 'uuid' not in item:
            return  # noqa
        return Zone(item['zone'], item['uuid'], item['type'])

    def records(self, zone_uuid, name, ztypes):
        # type: (str, str, str) -> [Record]
        offset = 0
        result = []

        while True:
            data = self._fetch('zones/{}/records?showRecords&limit=100&offset={}'.format(zone_uuid, offset))
            for item in data['zones'][0]['recordsList']['records']:
                if item['left-side'].rstrip('.') != name.rstrip('.'):
                    continue
                if item['type'] not in ztypes:
                    continue

                rec = Record(item['type'], item['ttl'], item['left-side'], item['right-side'])
                result.append(rec)

            total_count = data['zones'][0]['recordsList']['totalEntries']
            offset += len(data['zones'][0]['recordsList']['records'])
            if offset >= total_count:
                break

        return result

    def delete_record(self, record):
        # type: (Record) -> dict
        payload = {
            'operation': 'delete',
            'type': record.type,
            'name': record.left,
            'data': record.right,
            'ttl': int(record.ttl),
        }
        return self._put([payload])

    def add_record(self, record):
        # type: (Record) -> dict
        payload = {
            'operation': 'add',
            'type': record.type,
            'name': record.left,
            'data': record.right,
            'ttl': int(record.ttl),
        }
        return self._put([payload])

    def add_cname(self, left, right):
        # type: (str, str) -> dict
        return self.add_record(Record('CNAME', 600, self.format_fqdn(left), self.format_fqdn(right)))

    def delete_cname(self, left, right):
        # type: (str, str) -> dict
        return self.delete_record(Record('CNAME', 600, self.format_fqdn(left), self.format_fqdn(right)))

    def format_fqdn(self, fqdn):
        if not fqdn.endswith('.'):
            fqdn += '.'
        return fqdn
