# coding: utf-8
import logging

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.task import Status
from sandbox.projects.common import link_builder as lb
from sandbox.projects.music.deployment.MusicWatchDeployment import MusicWatchDeployment
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.projects.music.deployment.helpers.Nyan import nyan
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper, subtaskable
from sandbox.projects.release_machine.tasks.RollbackCommit import RollbackCommit
import sandbox.common.types.misc as ctm


class MusicRevertCommit(MusicBaseTask, TaskHelper):
    """ Roll back commits from music trunk or branch """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        description = "Roll back commits from music trunk or branch"

        kill_timeout = 60

        with sdk2.parameters.String("Revert mode", default="Trunk") as mode:
            mode.values["Trunk"] = mode.Value(value="Trunk")
            mode.values["Branch"] = mode.Value(value="Branch")

        with mode.value["Branch"]:
            branch = sdk2.parameters.String("Branch name", description="stable-01", default="", required=True)

        commit = sdk2.parameters.Integer("Commits to revert", description="100, r200, 300  400", required=True)
        debug = sdk2.parameters.Bool("Turn on debug mode", description="Sends message to stepanar@", default=False)

    def template_success(self, mode, initial_commit, rollback_task):
        logging.info("Mode is {}".format(mode))
        if mode == "Trunk":
            branch = mode
        else:
            branch = self.Parameters.branch

        merged_revs = ""
        for rev in rollback_task.Context.rolled_back_revs:
            merged_revs += "[{commit}]({commit_url})".format(commit=rev, commit_url=CONFIG.arc_commit_url(rev))

        # https://emojipedia.org/white-heavy-check-mark/
        check_mark = "\U00002705".decode("unicode-escape")
        template = MusicWatchDeployment.preprocess_template("""
        {check_mark} Реверт коммита [{commit}]({commit_url}) в {branch} прошел успешно
        Задача в сендбоксе: [{sandbox_task}]({sandbox_url})
        Дочерняя задача: [{child_task}]({child_url})
        Итоговый коммит:
        {merged_revs}

        По просьбе: {author}
        """)
        return template.format(commit=initial_commit,
                               commit_url=CONFIG.arc_commit_url(initial_commit),
                               merged_revs=merged_revs,
                               check_mark=check_mark,
                               branch=branch,
                               sandbox_task=self.id,
                               sandbox_url=lb.task_link(self.id, plain=True),
                               child_task=rollback_task.id,
                               child_url=lb.task_link(rollback_task.id, plain=True),
                               author=self.author)

    def template_failed(self, mode, commit, rollback_task):
        logging.info("Mode is {}".format(mode))
        if mode == "Trunk":
            branch = mode
        else:
            branch = self.Parameters.branch

        # \U0001F6D1 is a stop sing emoji https://emojipedia.org/octagonal-sign/
        stop_sign = "\U0001F6D1".decode("unicode-escape")
        template = MusicWatchDeployment.preprocess_template("""
        {stop_sign} Реверт коммита [{commit}]({commit_url}) в {branch} сломался
        Задача в сендбоксе: [{sandbox_task}]({sandbox_url})
        Дочерняя задача: [{child_task}]({child_url})

        По просьбе: {author}
        """)

        return template.format(commit=commit,
                               commit_url=CONFIG.arc_commit_url(commit),
                               branch=branch,
                               author=self.author,
                               stop_sign=stop_sign,
                               sandbox_task=self.id,
                               sandbox_url=lb.task_link(self.id, plain=True),
                               child_task=rollback_task.id,
                               child_url=lb.task_link(rollback_task.id, plain=True)
                               )

    @subtaskable(True)
    def rollback(self, branch, commit):
        logging.info("Branch {}", branch)
        if branch == "Trunk":
            sub_task = RollbackCommit(
                self,
                description="Child of {} task {}".format(self.parent, self.id),
                notifications=self.Parameters.notifications,
                rollback_mode="trunk",
                revs=commit,
                create_sub_task=True
            )
        else:
            sub_task = RollbackCommit(
                self,
                description="Child of {} task {}".format(self.parent, self.id),
                notifications=self.Parameters.notifications,
                rollback_mode="custom",
                revs=commit,
                branch_path=branch,
                create_sub_task=True
            )
        sub_task.enqueue()
        raise sdk2.WaitTask(sub_task, Status.Group.FINISH)

    @subtaskable(False)
    def check_subtask(self, rollback_commit):
        mode = self.Parameters.mode

        if mode == "Trunk":
            branch = mode
        else:
            branch = self.Parameters.branch

        logging.info("Branch: ", self.Parameters.branch)
        commit = self.Parameters.commit

        if rollback_commit.status == Status.SUCCESS:
            success_msg = self.template_success(branch, commit, rollback_commit)
            if self.Parameters.debug:
                nyan(success_msg, chat_id=CONFIG.stepanar_tg_chat)
            else:
                nyan(success_msg)
            return
        else:
            failed_msg = self.template_failed(branch, commit, rollback_commit)
            if self.Parameters.debug:
                nyan(failed_msg, chat_id=CONFIG.stepanar_tg_chat)
            else:
                nyan(failed_msg)

            msg = "Subtask {} failed with status {}".format(rollback_commit.id, rollback_commit.status)
            raise errors.TaskFailure(msg)

    def on_execute(self):
        self.check_authorization(self.author, CONFIG.token, CONFIG.auth_build)
        branch_prefix = "/branches/music/%s"
        mode = self.Parameters.mode
        commit = self.Parameters.commit

        if mode == "Trunk":
            self.rollback(mode, commit)
        elif mode == "Branch":
            self.rollback(branch_prefix % self.Parameters.branch, commit)
        else:
            raise errors.TaskError("Unknown mode {}".format(mode))

        self.check_subtask()
