# coding: utf-8
import logging
import re
from typing import Type, cast

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.sdk2 as sdk2
from sandbox.common import errors
from sandbox.projects.music.deployment.MusicBuildJars import MusicBuildJars
from sandbox.projects.music.deployment.MusicCheckRevisionIsGreen import MusicCheckRevisionIsGreen
from sandbox.projects.music.deployment.MusicUpdateNannyRevision import MusicUpdateNannyRevision
from sandbox.projects.music.deployment.MusicUpdateOpenapiRelease import MusicUpdateOpenapiRelease
from sandbox.projects.music.deployment.MusicWatchDeployment import MusicWatchDeployment
from sandbox.projects.music.deployment.helpers.AbcHelper import AbcHelper
from sandbox.projects.music.deployment.helpers.ArcadiaHelper import ArcadiaHelper
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.Deployment import Deployment
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.projects.music.deployment.helpers.Nyan import nyan
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper, subtaskable
from sandbox.projects.release_machine.tasks.CreateBranchOrTag import CreateBranchOrTag

TRUNK = 'trunk'
URL_RE = re.compile(r'^arcadia:/arc/(?:trunk|branches/music/(stable-\d+))/arcadia(?:@(\d+))?$')
START_WITH_ISSUE_RE = re.compile(r'^[A-Z]{2,}-\d+')


class MusicStartDeployment(MusicBaseTask, TaskHelper):
    """ Start Music deployment, instead of the testenv """
    fail_on_any_error = True

    class Requirements(cast(Type[sdk2.task.Requirements], sdk2.Task.Requirements)):
        environments = [TaskHelper.startrek_client_environment]
        cores = 1
        dns = ctm.DnsType.DNS64

        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name="music_deployment")
            ],
            release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(cast(Type[sdk2.task.Parameters], sdk2.Task.Parameters)):
        kill_timeout = 10 * 60  # real time ~1 min
        description = "Build and deploy Music"

        with sdk2.parameters.RadioGroup("Deployment mode") as mode:
            mode.values['standard'] = mode.Value('Standard: make a branch, build, deploy to testing and QA')
            mode.values['hotfix'] = mode.Value('Hotfix: build and deploy to stable and prestable')
            mode.values['prestable'] = mode.Value('Release: deploy to prestable')
            mode.values['stable'] = mode.Value('Release: deploy to stable')
            mode.values['rollback'] = mode.Value('Rollback: deploy previous nanny snapshot')

        with mode.value['standard']:
            # also: self.Context.url <- url with a revision set
            standard_url = sdk2.parameters.String("URL to build from",
                                                  default=CONFIG.arcadia_trunk,
                                                  description=CONFIG.arcadia_description,
                                                  required=True)
            standard_skip_checks = sdk2.parameters.Bool("Skip checks",
                                                        description="!!This will break SOX compatibility unless you "
                                                                    "add checks manually afterwards!!",
                                                        required=True,
                                                        default=False)

        with mode.value['hotfix']:
            # also: self.Context.url <- url with a revision set
            hotfix_url = sdk2.parameters.String("URL to build from",
                                                default=CONFIG.arcadia_branch.format('XXX'),
                                                description=CONFIG.arcadia_description,
                                                required=True)
            with sdk2.parameters.CheckGroup("Component", required=True) as hotfix_components:
                for name in CONFIG.components:
                    setattr(hotfix_components.values, name, hotfix_components.Value(name))

            hotfix_reason = sdk2.parameters.String("The reason for this hotfix",
                                                   default="",
                                                   description="MUSICBACKEND-1234 Text reason here",
                                                   required=True)

        with mode.value['prestable']:
            prestable_task = sdk2.parameters.Task("{} task".format(CONFIG.build_jars_task_type),
                                                  task_type=CONFIG.build_jars_task_type)

        with mode.value['stable']:
            stable_task = sdk2.parameters.Task("{} task".format(CONFIG.build_jars_task_type),
                                               task_type=CONFIG.build_jars_task_type)
        with mode.value['rollback']:
            rollback_reason = sdk2.parameters.String("Ticket reason for rollback",
                                                     default="",
                                                     description="MUSICBACKEND-1234",

                                                     required=True)

            st_report_ticket = sdk2.parameters.String(
                "MUSICRELEASES ticket to report to",
                description=("Report to startreck ticket about this rollback. "
                             "By default will report to open MUSICRELEASES ticket "),
                required=False
            )

            with sdk2.parameters.RadioGroup("Rollback env") as rollback_env:
                rollback_env.values.testing = mode.Value('Testing: rollback only testing environment')
                rollback_env.values.qa = mode.Value('Qa: rollback only qa environment')
                rollback_env.values.prestable = mode.Value('Prestable: rollback only prestable environment')
                rollback_env.values.stable = mode.Value('Stable: rollback only stable environment')
                rollback_env.values.prod = mode.Value(
                    'Stable + prestable: rollback stable and prestable environments')

            with sdk2.parameters.CheckGroup("Limit to components", required=False) as components:
                for name in CONFIG.components:
                    setattr(components.values, name, components.Value(name))

        with sdk2.parameters.Output():
            branch = sdk2.parameters.String("The branch name")
            revision = sdk2.parameters.Integer("Build revision")
            issue = sdk2.parameters.String("Startrek issue")

        abc_token = sdk2.parameters.YavSecret("ABC OAuth token secret", required=True,
                                              description="Yav secret with ABC token",
                                              default='sec-01fp7s63sd8dgjga0erprk7tfx')

        abc_token_key = sdk2.parameters.String("Key for abc token in yav secret", required=True,
                                               default="abc_token")

    class Context(cast(Type[sdk2.task.Context], sdk2.Task.Context)):
        pass

    @subtaskable(True)
    def process_input_parameters_standard(self, token):
        match = URL_RE.match(str(self.Parameters.standard_url))
        if not match:
            raise errors.TaskFailure('The URL is in incompatible format')
        branch, revision = match.groups()

        if revision:
            self.Context.revision = int(revision)
            self.Context.url = str(self.Parameters.standard_url)
        else:
            self.Context.revision = ArcadiaHelper.get_latest_affected_revision(self.Parameters.standard_url + '/music')
            self.Context.url = '{}@{}'.format(self.Parameters.standard_url, self.Context.revision)

        self.Context.branch = branch or TRUNK

        st = StartrekHelper(token)
        if self.Context.branch == TRUNK:
            if len(st.find_nonclosed_issues()) > 0:
                raise errors.TaskFailure('Cannot create a new release while there is a pending release')

    @subtaskable(True)
    def process_input_parameters_hotfix(self, token):
        if not self.Parameters.hotfix_reason or not START_WITH_ISSUE_RE.match(str(self.Parameters.hotfix_reason)):
            raise errors.TaskFailure('Please specify a reason for this hotfix with a startrek issue')

        match = URL_RE.match(str(self.Parameters.hotfix_url))
        if not match:
            raise errors.TaskFailure('The url is in incompatible format')
        branch, revision = match.groups()

        if not branch:
            raise errors.TaskFailure('Hotfixes can only be deployed from branches')

        if not revision:
            revision = ArcadiaHelper.get_latest_affected_revision(self.Parameters.hotfix_url + '/music')

        if not list(self.Parameters.hotfix_components):
            raise errors.TaskFailure('A hotfix must have at least one component set')

        self.Context.branch = branch
        self.Context.revision = int(revision)
        self.Context.url = self.Parameters.hotfix_url

        st = StartrekHelper(token)
        issue = st.find_issue_for_branch(self.Context.branch)
        if not issue:
            raise errors.TaskFailure('Hotfix issue not found')

    @subtaskable(True)
    def process_input_parameters_stable(self):
        if str(self.Parameters.stable_task.type) != CONFIG.build_jars_task_type:
            raise errors.TaskFailure('The task type must be a {}, not {}'.format(CONFIG.build_jars_task_type,
                                                                                 self.Parameters.stable_task.type))

        self.Context.branch = self.Parameters.stable_task.Parameters.branch
        self.Context.revision = self.Parameters.stable_task.Parameters.revision
        self.Context.url = 'arcadia:/arc/branches/music/{}/arcadia@{}'.format(self.Context.branch,
                                                                              self.Context.revision)

    @subtaskable(True)
    def process_input_parameters_prestable(self):
        if str(self.Parameters.prestable_task.type) != CONFIG.build_jars_task_type:
            raise errors.TaskFailure('The task type must be a {}, not {}'.format(CONFIG.build_jars_task_type,
                                                                                 self.Parameters.prestable_task.type))

        self.Context.branch = self.Parameters.prestable_task.Parameters.branch
        self.Context.revision = self.Parameters.prestable_task.Parameters.revision
        self.Context.url = 'arcadia:/arc/branches/music/{}/arcadia@{}'.format(self.Context.branch,
                                                                              self.Context.revision)

    @subtaskable(True)
    def check_revision_is_green(self):
        branch_name = 'music/' + str(self.Context.branch) if self.Context.branch != TRUNK else TRUNK

        if self.Context.branch == TRUNK and not self.Parameters.standard_skip_checks:
            self.enqueue_subtask(MusicCheckRevisionIsGreen,
                                 revision=self.Context.revision,
                                 branch=branch_name,
                                 description='Checking {}'.format(self.Context.url))

    @subtaskable(True)
    def create_branch(self, music_check_revision_is_green):
        if self.Context.branch == TRUNK:
            nyan(u'⚙️ Пришло время нового релиза! Буду собирать из ревизии {}'.format(self.Context.revision))
            self.enqueue_subtask(CreateBranchOrTag,
                                 component_name='music',
                                 tag_or_branch_mode='trunk->branch',
                                 revision_for_trunk=self.Context.revision,
                                 description='Branching from revision {}'.format(self.Context.revision))
        else:
            nyan(u'⚙️ Собираю {}/{} по просьбе {}'.format(self.Context.branch, self.Context.revision, self.author))

    @subtaskable(True)
    def create_startrek_issue(self, token, create_branch_or_tag):
        startrek = StartrekHelper(token)
        if create_branch_or_tag:
            self.Context.branch = create_branch_or_tag.Parameters.result_path.split('/')[-1]
            self.Context.revision = create_branch_or_tag.Parameters.result_revision

            # update build url to the new version
            branch_no = self.Context.branch.split('-')[1]
            self.Context.url = CONFIG.arcadia_branch.format(branch_no) + '@' + str(self.Context.revision)

            arcadia = ArcadiaHelper(startrek)
            parsed_changelog, commits_without_task, first_rev, _ = arcadia.prepare_and_extract_latest_changelog(
                self.Context.branch)
            on_duty = self.backend_on_duty()
            text = arcadia.render_changelog(parsed_changelog=parsed_changelog,
                                            commits_without_tasks=commits_without_task,
                                            from_revision=first_rev,
                                            to_revision=self.Context.revision,
                                            branch=self.Context.branch,
                                            on_duty=on_duty)
            issue = startrek.create_issue(
                CONFIG.issue_summary(self.Context.revision, self.Context.branch),
                text,
                CONFIG.checklist,
                assignee=on_duty,
                qa_engineer=self.qa_on_duty())
        else:
            issue = startrek.find_issue_for_branch(self.Context.branch)
        self.Context.issue = issue.key
        return issue

    @subtaskable(True)
    def build(self):
        task = Deployment.find_build_task(self.Context.revision, self.Context.branch)
        if task:
            if str(task.type) != CONFIG.build_jars_task_type or \
                    task.Parameters.branch != self.Context.branch or \
                    task.Parameters.revision != self.Context.revision or \
                    task.Parameters.patch:
                logging.error('I was given task {}! Falling back to rebuild!'.format(task.id))
            else:
                self.Context.done_task = task.id

        if not self.Context.done_task:
            self.enqueue_subtask(MusicBuildJars,
                                 url=self.Context.url,
                                 description='Building {}'.format(self.Context.url),
                                 jdk_versions=[CONFIG._DEFAULT_JDK_VERSION])

    @subtaskable(True)
    def initiate_deployment_standard(self, music_build_jars):
        self.enqueue_subtask(MusicUpdateNannyRevision,
                             task=music_build_jars if music_build_jars else sdk2.Task[self.Context.done_task],
                             issue=self.Context.issue,
                             target=['qa', 'testing'],
                             description='Initiating {} deployment'.format(self.Context.url))

    @subtaskable(True)
    def initiate_deployment_hotfix(self, token, music_build_jars):
        st = StartrekHelper(token)
        self.Context.issue = st.find_issue_for_branch(self.Context.branch).key

        self.enqueue_subtask(MusicUpdateNannyRevision,
                             task=music_build_jars if music_build_jars else sdk2.Task[self.Context.done_task],
                             issue=self.Context.issue,
                             target=['prestable', 'stable'],
                             components=self.Parameters.hotfix_components,
                             description='Initiating {} deployment'.format(self.Context.url))

    @subtaskable(True)
    def initiate_deployment_stable(self, token):
        st = StartrekHelper(token)
        self.Context.issue = st.find_issue_for_branch(self.Context.branch).key

        self.enqueue_subtask(MusicUpdateNannyRevision,
                             task=self.Parameters.stable_task,
                             issue=self.Context.issue,
                             target=['prestable', 'stable'],
                             description='Initiating {} deployment'.format(self.Context.url))

    @subtaskable(True)
    def initiate_deployment_prestable(self, token):
        st = StartrekHelper(token)
        self.Context.issue = st.find_issue_for_branch(self.Context.branch).key

        self.enqueue_subtask(MusicUpdateNannyRevision,
                             task=self.Parameters.prestable_task,
                             issue=self.Context.issue,
                             target=['prestable'],
                             description='Initiating {} deployment'.format(self.Context.url))

    @subtaskable(False)
    def initiate_openapi_spec_deployment(self):
        self.enqueue_subtask(
            MusicUpdateOpenapiRelease,
            description='Music release (Ticket: {}, Branch: {})'.format(self.Context.issue, self.Context.branch)
        )

    @subtaskable(True)
    def initiate_rollback(self, token):

        if not self.Parameters.st_report_ticket:
            st = StartrekHelper(token)
            issue = st.find_nonclosed_issues()
            self.Context.issue = issue[0].key
        else:
            self.Context.issue = self.Parameters.st_report_ticket

        if self.Parameters.rollback_env == 'prod':
            target = ['stable', 'prestable']
        else:
            target = [self.Parameters.rollback_env]

        self.enqueue_subtask(MusicUpdateNannyRevision,
                             issue=self.Context.issue,
                             mode='rollback',
                             target=target,
                             components=self.Parameters.components,
                             description='Initiating rollback')

    @subtaskable(True)
    def finalize_rollback(self, token, music_update_nanny_revision):
        self.Context.branch = music_update_nanny_revision.Context.branch
        self.Context.revision = music_update_nanny_revision.Context.revision
        self.Context.url = music_update_nanny_revision.Context.url

        if self.Parameters.rollback_env in ['stable', 'prestable', 'prod']:
            self.mark_rollback_finilazation_in_startrek(token,
                                                        music_update_nanny_revision.Parameters.target,
                                                        music_update_nanny_revision.Parameters.components,
                                                        music_update_nanny_revision.id,
                                                        self.Parameters.rollback_reason)

            self.enqueue_watcher(music_update_nanny_revision, 'rollback_' + self.Parameters.rollback_env,
                                 str(self.Parameters.rollback_reason))

    def mark_rollback_finilazation_in_startrek(self, token, target, components, deploy_task, reason=None):
        st = StartrekHelper(token)
        text = (u'Запущен ОТКАТ релиза\n\n'  # this line must not change. Used for statistics generation.
                u'Окружения: {target}\n\n'
                u'Компоненты: {components}\n\n'
                u'Ревизия, на которую откатываемся:\n'
                u'Ветка: {branch}\n'
                u'Ревизия: {revision}\n'
                u'Выкладочная задача: {deploy_task}\n').format(
            target=', '.join(target),
            components=', '.join(components),
            branch=self.Context.branch,
            revision=self.Context.revision,
            deploy_task=CONFIG.wikiformat_sandbox(str(deploy_task)),
        )

        if reason:
            # the reason always comes in "TICKET-XXX Text" form, and https is required
            # for kate2110@ to generate statistics
            text += u'\n\nПричина отката: https://st.yandex-team.ru/{}'.format(reason)

        st.add_comment(self.Context.issue, text)

    def extract_hotfix_changelog(self, startrek):
        arcadia = ArcadiaHelper(startrek)
        issue = startrek.get_issue(self.Context.issue)
        revision, branch = startrek.summary_revision_and_branch(issue.summary)
        path = arcadia.get_url(self.Context.branch, subdir='arcadia/music')
        tasks, commits, last_rev = arcadia.extract_changelog(path, revision)
        previous_branch = arcadia.find_previous_branch(self.Context.branch)
        logging.info('Current branch: {}, previous: {}'.format(self.Context.branch, previous_branch))
        first_rev = arcadia.branch_to_revision(previous_branch)
        logging.info('Render hotfix changelog r%s:r%s', first_rev, last_rev)
        text = arcadia.render_changelog(parsed_changelog=tasks,
                                        commits_without_tasks=commits,
                                        from_revision=first_rev,
                                        to_revision=last_rev,
                                        branch=branch,
                                        on_duty=self.backend_on_duty(),
                                        hotfix=True)
        return text

    def mark_finalization_in_startrek(self, token, target, build_task, deploy_task, changelog=False, reason=None):
        st = StartrekHelper(token)
        text = (u'Запущена выкатка {target}\n\n'
                u'Ветка: {branch}\n'
                u'Ревизия: {revision}\n'
                u'Сборочная задача: {build_task}\n'
                u'Выкладочная задача: {deploy_task}\n'
                u'Тесты: (({ci_url} CI))\n').format(
            target=target,
            branch=self.Context.branch,
            revision=self.Context.revision,
            build_task=CONFIG.wikiformat_sandbox(str(build_task)),
            deploy_task=CONFIG.wikiformat_sandbox(str(deploy_task)),
            ci_url=Deployment.get_ci_url(self.Context.revision, 'music/' + self.Context.branch),
        )

        if changelog:
            text += "\n" + self.extract_hotfix_changelog(st)

        if reason:
            # the reason always comes in "TICKET-XXX Text" form, and https is required
            # for kate2110@ to generate statistics
            text += u'\n\nПричина выкатки: https://st.yandex-team.ru/{}'.format(reason)

        st.add_comment(self.Context.issue, text)
        st.set_issue_release_notes(self.Context.issue, str(build_task))

    def enqueue_watcher(self, music_update_nanny_revision, target, reason=''):
        self.enqueue_subtask(MusicWatchDeployment,
                             dashboards=music_update_nanny_revision.Parameters.dashboards,
                             branch=self.Context.branch,
                             revision=self.Context.revision,
                             issue=self.Context.issue,
                             target=target,
                             reason=reason)

    @subtaskable(True)
    def finalize_standard(self, token, music_update_nanny_revision, music_build_jars):
        task_id = music_build_jars.id if music_build_jars else self.Context.done_task
        self.mark_finalization_in_startrek(token,
                                           u'в тестинг и QA',
                                           task_id,
                                           music_update_nanny_revision.id)
        self.enqueue_watcher(music_update_nanny_revision, 'testing')

    @subtaskable(True)
    def finalize_hotfix(self, token, music_update_nanny_revision, music_build_jars):
        task_id = music_build_jars.id if music_build_jars else self.Context.done_task
        self.mark_finalization_in_startrek(token,
                                           u'ХОТФИКСА',
                                           task_id,
                                           music_update_nanny_revision.id,
                                           changelog=True,
                                           reason=self.Parameters.hotfix_reason)
        self.enqueue_watcher(music_update_nanny_revision, 'hotfix', str(self.Parameters.hotfix_reason))

    @subtaskable(True)
    def finalize_stable(self, token, music_update_nanny_revision):
        task_id = self.Parameters.stable_task.id
        self.mark_finalization_in_startrek(token,
                                           u'в бой',
                                           task_id,
                                           music_update_nanny_revision.id)
        self.enqueue_watcher(music_update_nanny_revision, 'stable')

    @subtaskable(True)
    def finalize_prestable(self, token, music_update_nanny_revision):
        task_id = self.Parameters.prestable_task.id
        self.mark_finalization_in_startrek(token,
                                           u'в prestable',
                                           task_id,
                                           music_update_nanny_revision.id)
        self.enqueue_watcher(music_update_nanny_revision, 'prestable')

    def on_execute(self):

        abc_token = self.Parameters.abc_token.data()[self.Parameters.abc_token_key]
        self.abc_helper = AbcHelper(abc_token)

        self.check_authorization(self.author, CONFIG.token, CONFIG.auth_build, force_duty=True)
        self.check_there_is_only_one_such_task()

        token = sdk2.Vault.data(CONFIG.token)
        if self.Parameters.mode == 'standard':
            self.process_input_parameters_standard(token)
            self.check_revision_is_green()
            self.create_branch()
            self.create_startrek_issue(token)
            self.build()
            self.initiate_deployment_standard()
            self.finalize_standard(token)
        elif self.Parameters.mode == 'hotfix':
            self.process_input_parameters_hotfix(token)
            self.build()
            self.initiate_deployment_hotfix(token)
            self.finalize_hotfix(token)
        elif self.Parameters.mode == 'prestable':
            self.process_input_parameters_prestable()
            self.initiate_deployment_prestable(token)
            self.finalize_prestable(token)
        elif self.Parameters.mode == 'stable':
            self.process_input_parameters_stable()
            self.initiate_deployment_stable(token)
            self.finalize_stable(token)
            self.initiate_openapi_spec_deployment()
        elif self.Parameters.mode == "rollback":
            self.initiate_rollback(token)
            self.finalize_rollback(token)
        else:
            raise errors.TaskError('Unknown mode {}'.format(self.Parameters.mode))

        self.Parameters.branch = cast(sdk2.parameters.String, self.Context.branch)
        self.Parameters.revision = self.Context.revision
        self.Parameters.issue = self.Context.issue
