# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging as log
import re

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import task as ctt
from sandbox.common.types.task import Status
from sandbox.projects.common import link_builder as lb
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper
from sandbox.projects.music.resources import MusicTankAmmo
from sandbox.projects.tank.LoadTestResults.loadtest_report import ReportLoadTestResults
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi

TANK_CONFIG_TMPL = """
phantom:
    timeout: "4"
    load_profile:
        load_type: rps
        schedule: step(100, 400, 50, 5s) const(400,10m)
    autocases: 1
telegraf:
    enabled: true
uploader:
    job_name: 'testing tank automation'
    operator: robot-music-admin
    ver: "{release_ticket}"
    task: {release_ticket}
    component: music_backend_regression
"""

MONITORING_CONFIG = """
<Monitoring>
    <Host address="[target]" interval="1" username="robot-music-admin">
        <CPU measure="idle,user,system,iowait" />
        <System measure="la1,csw,int,numproc,numthreads" />
        <Memory measure="free,used,cached,buff" />
        <Disk measure="read,write"/>
        <Net measure="recv,send,retransmit,estab,closewait,timewait"/>
    </Host>
    <Host address="localhost">
        <CPU measure="idle,user,system,iowait" />
        <System measure="la1,csw,int,numproc,numthreads" />
        <Memory measure="free,used,cached,buff" />
        <Disk measure="read,write"/>
        <Net measure="recv,send,retransmit,estab,closewait,timewait"/>
    </Host>
</Monitoring>
"""

MINUTE = 60


class MusicTankRegressionTest(MusicBaseTask, TaskHelper):
    class Requirements(sdk2.Task.Requirements):
        environments = [TaskHelper.startrek_client_environment]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60

        description = "Load test music backend api"

        nanny_tank_target = sdk2.parameters.String("Tank target Nanny service", required=True,
                                                   default="music_load_back_sas")
        tank_to_use = sdk2.parameters.String("Tank to use for shoot (should be in the same DC)", required=True,
                                             default="nanny:music_tanks_sas")
        issue = sdk2.parameters.String("Music backend release issue", required=True)

        extra_message = sdk2.parameters.String("Extra message", required=False)  # type: str

        report_to_st = sdk2.parameters.Bool("Send report to startrek issue", default=True)  # type: bool

        sleep_on_start = sdk2.parameters.Integer("Minutes to delay before task starts", default=0)  # type: int

        with sdk2.parameters.Group('Ammo parameters') as ammo_block:
            use_latest_ammo = sdk2.parameters.Bool("Use latest ammo resource", default=True)
            with use_latest_ammo.value[False]:
                tank_ammo = sdk2.parameters.Resource("Resource with ammo", required=False)

    class Context(sdk2.Task.Context):
        lunapark_link = None
        desc = "Music backend stress test"
        tank_subtasks = None
        report_tasks = []

    def _start_shooting(self):
        log.info("Starting to shoot")
        ctx = self.Context

        tank_ammo = self.Parameters.tank_ammo if self.Parameters.tank_ammo else MusicTankAmmo.find().first()
        log.info("Tank ammo {}".format(tank_ammo))
        try:
            shoot_task = ShootViaTankapi(
                self,
                nanny_service=self.Parameters.nanny_tank_target,
                override_nanny_port="80",
                use_public_tanks=False,
                tanks=self.Parameters.tank_to_use,
                description=ctx.desc,
                use_monitoring=True,
                ammo_source="resource",
                ammo_resource=tank_ammo,
                config_source="file",
                config_content=TANK_CONFIG_TMPL.format(release_ticket=self.Parameters.issue),
                monitoring_source="file",
                monitoring_content=MONITORING_CONFIG
            ).enqueue()
        except Exception as ex:
            log.exception("Could not call subtask")
            raise errors.TaskFailure(ex)
        else:
            ctx.tank_subtask = shoot_task.id
            raise sdk2.WaitTask(
                [ctx.tank_subtask],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True,
                timeout=1800)

    def start_reporting(self):
        token = sdk2.Vault.data(CONFIG.token)

        ctx = self.Context
        st = StartrekHelper(token)
        st_queue = self.Parameters.issue.split('-')[0].lower()

        if not ctx.lunapark_link:
            return
        log.info('Lunapark link for shoot is %s', ctx.lunapark_link)
        shoot_id = re.findall(r'\d+', ctx.lunapark_link)[0]
        try:
            report = ReportLoadTestResults(shoot_id=shoot_id, report_type='const')
        except Exception as ex:
            # TODO: add summonees=self.backend_on_duty() when this is ready for production
            st.add_comment(self.Parameters.issue, ex)
            raise

        report.get_summary()
        report.empty_test = True  # must leave it here or else message will contain some junk
        report.build_message()

        failed = False  # this comes from api. ¯\_(ツ)_/¯
        if report.result["regression"]["resolution"] == failed:
            report.message = "\n\nЗадача танковых стрельб !!сломалась!! " \
                             "((https://sandbox.yandex-team.ru/task/{0} {0}))\n".format(self.id) + \
                             report.message + \
                             "\n\nСcылка на регресс:\n" + \
                             "https://lunapark.yandex-team.ru/regress/{}#music_backend_regression".format(st_queue)
        else:
            report.message = "\n\nТанковые стрельбы прошли успешно  {}\n".format(self.id) + \
                             report.message

        if self.Parameters.extra_message:
            report.message += "\n" + self.Parameters.extra_message

        st.add_comment(self.Parameters.issue, report.message)
        if report.result["regression"]["resolution"] == failed:
            raise errors.TaskFailure(report.message)
        else:
            st.patch_release_issue_statuses_raw(self.Parameters.issue, CONFIG.monitored_deployment_test_names,
                                                self.__class__.__name__)

    def check_subtask_status(self, task_id):
        token = sdk2.Vault.data(CONFIG.token)
        task = sdk2.Task[task_id]
        if task.status != Status.SUCCESS:
            task_link = lb.task_link(task.id, plain=True)
            st = StartrekHelper(token)
            st.add_comment(self.Parameters.issue, "Task failed\n{}".format(task_link))
            raise errors.TaskFailure(task.info)

    def on_enqueue(self):
        with self.memoize_stage.sleep:
            if self.Parameters.sleep_on_start > 0:
                raise sdk2.WaitTime(self.Parameters.sleep_on_start * MINUTE)

    def on_execute(self):
        ctx = self.Context

        with self.memoize_stage.shooting:
            self._start_shooting()

        self.check_subtask_status(ctx.tank_subtask)
        ctx.lunapark_link = sdk2.Task[ctx.tank_subtask].Parameters.lunapark_link
        #
        if self.Parameters.report_to_st:
            self.start_reporting()
