# coding: utf-8
import json
import logging
import re

import requests

import sandbox.sdk2 as sdk2
from sandbox.common import errors
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.DashboardUpdater import DashboardUpdater
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper


class MusicUpdateNannyRevision(MusicBaseTask, TaskHelper):
    """ Update Music Nanny dashboards to the corresponding revision """
    fail_on_any_error = True
    _issue_re = re.compile(r'^MUSICRELEASES-\d+$')

    class Requirements(sdk2.Task.Requirements):
        environments = [TaskHelper.startrek_client_environment]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 60 * 60  # nanny is really slow nowadays

        task = sdk2.parameters.Task("The {} task".format(CONFIG.build_jars_task_type),
                                    required=False,
                                    task_type=CONFIG.build_jars_task_type)
        issue = sdk2.parameters.String("The release startrek issue", required=True)

        with sdk2.parameters.RadioGroup("Deployment mode") as mode:
            mode.values['standard'] = mode.Value('Standard: update revision and deploy dashboard',
                                                 default=True)
            mode.values['rollback'] = mode.Value('Rollback: deploy previous nanny snapshot')

        with sdk2.parameters.CheckGroup("Deployment target(s)", required=True) as target:
            target.values.testing = "testing"
            target.values.qa = "qa"
            target.values.prestable = "prestable"
            target.values.stable = "stable"

        with sdk2.parameters.CheckGroup("Limit to components", required=False) as components:
            for name in CONFIG.components:
                setattr(components.values, name, components.Value(name))

        with sdk2.parameters.Output():
            dashboards = sdk2.parameters.String("Running dashboard ids")
            services_to_sandbox_task = sdk2.parameters.String("Nanny service to sandbox task")

    def _validate_issue(self, token):
        if not self._issue_re.match(str(self.Parameters.issue)):
            raise errors.TaskFailure('Bad issue number')

        st = StartrekHelper(token)
        issue = st.get_info(self.Parameters.issue)  # issue must exist

        if 'stable' in self.Parameters.target and issue['status'] == 'open':
            raise errors.TaskFailure('The issue {} is not in the releasing state'.format(self.Parameters.issue))

        if ('testing' in self.Parameters.target or 'qa' in self.Parameters.target) and \
                issue['status'] not in ('open', 'testingByAssessors', 'testing'):
            raise errors.TaskFailure(
                'The issue {} is not in the open/testing/testingByAssessors state'.format(self.Parameters.issue))

    def _update_revision(self, token):

        task_id = str(self.Parameters.task.id)
        recipe_names = [CONFIG.recipes[target] for target in self.Parameters.target]

        branch_and_revision = '{}/{}'.format(self.Parameters.task.Parameters.branch,
                                             self.Parameters.task.Parameters.revision)

        # do not raise on error.
        if 'stable' in self.Parameters.target:
            requests.post('https://gr-mg.yandex-team.ru/events/',
                          data=json.dumps(
                              {'what': 'Deploying {}'.format(branch_and_revision), 'tags': 'nyanbot.music.sre'}))

        if list(self.Parameters.components):
            dashboards = tuple('music-' + component for component in self.Parameters.components)
        else:
            dashboards = CONFIG.dashboards

        updater = DashboardUpdater(token)
        for dashboard in dashboards:
            updater.set_nanny_services(self.author,
                                       dashboard,
                                       recipe_names,
                                       task_id,
                                       branch_and_revision)

        result = {}
        for dashboard in dashboards:
            logging.info("Running dashboard {} recipe {}".format(dashboard, recipe_names))
            result[dashboard] = updater.run_dashboard(dashboard, recipe_names)

        self.Parameters.dashboards = json.dumps(result)

    def _rollback(self, token):
        recipe_names = [CONFIG.recipes[target] for target in self.Parameters.target]

        for t in self.Parameters.target:
            if t in ['stable', 'prestable']:
                requests.post('https://gr-mg.yandex-team.ru/events/',
                              data=json.dumps(
                                  {'what': 'Rollback {}'.format(self.Parameters.target), 'tags': 'nyanbot.music.sre'}))
            break  # report only once

        dashboards = CONFIG.dashboards
        if list(self.Parameters.components):
            dashboards = tuple('music-' + component for component in self.Parameters.components)

        updater = DashboardUpdater(token)
        svc_to_sandbox = {}
        for dashboard in dashboards:
            svc_to_sandbox.update(updater.set_rollback_revision(self.author,
                                                                dashboard,
                                                                recipe_names,
                                                                str(self.id)))

        unique_resources = set(svc_to_sandbox.values())
        if len(unique_resources) > 1:
            raise errors.TaskError("Multiple music backend versions for rollback. '{}'".format(unique_resources))
        else:
            music_build_jars = sdk2.Task.find(id=list(unique_resources)[0]).limit(1)
            for build in music_build_jars:
                logging.info("Build task {}".format(build))
                logging.info("Rollback branch {}".format(build.Parameters.branch))
                self.Context.branch = build.Parameters.branch
                logging.info("Rollback revision {}".format(build.Parameters.revision))
                self.Context.revision = build.Parameters.revision
                logging.info("Rollback url {}".format(build.Parameters.url))
                self.Context.url = build.Parameters.url

        result = {}
        for dashboard in dashboards:
            logging.info("Running dashboard {} recipe {}".format(dashboard, recipe_names))
            result[dashboard] = updater.run_dashboard(dashboard, recipe_names)

        self.Parameters.dashboards = json.dumps(result)
        self.Parameters.services_to_sandbox_task = json.dumps(svc_to_sandbox)

    def on_execute(self):
        self.check_authorization(self.author, CONFIG.token, CONFIG.auth_build)

        if not self.Parameters.target:
            raise errors.TaskFailure('At least one target must be specified')

        token = sdk2.Vault.data(CONFIG.token)
        self._validate_issue(token)

        if self.Parameters.mode == "standard":
            self._update_revision(token)
        elif self.Parameters.mode == "rollback":
            self._rollback(token)
