# coding: utf-8
import datetime
import json
import logging
from functools import partial
from textwrap import dedent

import requests

import sandbox.sdk2 as sdk2
from sandbox.common import errors
from sandbox.common.types.misc import NotExists, DnsType
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.music.deployment.MusicDeployTests import MusicDeployTests
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.projects.music.deployment.helpers.Nyan import nyan
from sandbox.projects.music.deployment.helpers.StartrekHelper import StartrekHelper
from sandbox.projects.music.deployment.helpers.TaskHelper import TaskHelper


class MusicWatchDeployment(MusicBaseTask, TaskHelper):
    """Watch deployment process and inform on success"""

    class Requirements(sdk2.Task.Requirements):
        environments = [
            TaskHelper.startrek_client_environment,
            TaskHelper.juggler_client_environment,
        ]
        cores = 1
        dns = DnsType.DNS64

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60
        description = "Watch deployment process and inform on success"

        dashboards = sdk2.parameters.String("Dashboard ids dict", required=True)
        branch = sdk2.parameters.String("The deployment branch", required=True)
        revision = sdk2.parameters.Integer("The deployment revision", required=True)
        issue = sdk2.parameters.String("The deployment issue", required=True)
        reason = sdk2.parameters.String("The deployment reason", default="", required=True)
        set_juggler_downtime = sdk2.parameters.Bool("Set juggler downtime for production deployment",
                                                    description="Sets juggler downtime for production deployment",
                                                    default=True,
                                                    required=True)

        with sdk2.parameters.RadioGroup("Deployment target", required=True) as target:
            target.values.testing = "testing"
            target.values.prestable = "prestable"
            target.values.stable = "stable"
            target.values.hotfix = "hotfix"
            target.values.rollback_prod = "rollback_prod"
            target.values.rollback_stable = "rollback_stable"
            target.values.rollback_prestable = "rollback_prestable"

    @staticmethod
    def dashboards_to_deployments(dashboards, environment):
        result = ''
        for dashboard in CONFIG.dashboards:
            recipe = 'deploy-' + environment
            if dashboard in CONFIG.dashboard_recipe_overrides:
                recipe = CONFIG.dashboard_recipe_overrides[dashboard].get(recipe, recipe)
            if dashboard in dashboards and recipe in dashboards[dashboard]:
                result += (
                    CONFIG.list_prefix +
                    u'[{}]({}){}\n'.format(
                        dashboard,
                        CONFIG.deployment_url(dashboard, dashboards[dashboard][recipe]['deployment']),
                        u' — готово' if dashboards[dashboard][recipe]['is_ready'] else ''
                    )
                )
        return result

    @staticmethod
    def preprocess_template(template):
        return '\n'.join([x.strip() for x in template.strip().split('\n')])

    def template_testing(self, deployments_testing, deployments_qa):
        template = self.preprocess_template(u"""
            🛶 Релиз {branch}/{revision} едет в тестинг и QA

            Тикет: {issue}

            Выкатки в тестинг:
            {deployments_testing}
            Выкатки в QA:
            {deployments_qa}
        """)
        return template.format(branch=self.Parameters.branch,
                               revision=self.Parameters.revision,
                               issue=CONFIG.markdown_startrek(self.Parameters.issue),
                               deployments_testing=deployments_testing,
                               deployments_qa=deployments_qa)

    def template_stable(self, stable, prestable):
        template = self.preprocess_template(u"""
            🚀 Релиз {branch}/{revision} едет в бой

            Тикет: {issue}

            Выкатки в prestable:
            {prestable}
            Выкатки в stable:
            {stable}
        """)
        return template.format(branch=self.Parameters.branch,
                               revision=self.Parameters.revision,
                               issue=CONFIG.markdown_startrek(self.Parameters.issue),
                               stable=stable,
                               prestable=prestable)

    def template_prestable(self, prestable):
        template = self.preprocess_template(u"""
            🚀 Релиз {branch}/{revision} едет в prestable

            Тикет: {issue}

            Выкатки:
            {prestable}
        """)
        return template.format(branch=self.Parameters.branch,
                               revision=self.Parameters.revision,
                               issue=CONFIG.markdown_startrek(self.Parameters.issue),
                               prestable=prestable)

    def template_hotfix(self, stable, prestable):
        template = self.preprocess_template(u"""
            🚀 Релиз {branch}/{revision} едет *ХОТФИКСОМ* в бой
            {reason}

            Тикет: {issue}

            Выкатки в prestable:
            {prestable}
            Выкатки в stable:
            {stable}
        """)
        return template.format(branch=self.Parameters.branch,
                               revision=self.Parameters.revision,
                               issue=CONFIG.markdown_startrek(self.Parameters.issue),
                               stable=stable,
                               prestable=prestable,
                               reason=self.Parameters.reason)

    def template_rollback(self, stable=None, prestable=None):
        template = dedent(u"""
            🚀 Релиз {branch}/{revision} едет *ОТКАТОМ* в бой
            {reason}

            Тикет: {issue}
        """.format(branch=self.Parameters.branch,
                   revision=self.Parameters.revision,
                   reason=self.Parameters.revision,
                   issue=CONFIG.markdown_startrek(self.Parameters.issue)))

        if prestable:
            template += dedent(u"""
                Выкатки в prestable:
                {}""").format(prestable)

        if stable:
            template += dedent(u"""
                Выкатки в stable:
                {}
                """).format(stable)

        return template

    def template_done(self, running_seconds):
        template = self.preprocess_template(u"""
            🎉 Выкатка {branch}/{revision} ({target}) завершена за {time} минут!
        """)
        return template.format(branch=self.Parameters.branch,
                               revision=self.Parameters.revision,
                               # telegram markdown mode fails if it doesn't find second '_'. e.g. 'rollback_prestable'
                               target=self.Parameters.target.replace("_", " "),
                               time=int(running_seconds / 60))

    def grafana_event_done(self):
        requests.post('https://gr-mg.yandex-team.ru/events/',
                      data=json.dumps(
                          {'what': 'Finished deploying {}/{}'
                              .format(self.Parameters.branch, self.Parameters.revision),
                           'tags': 'nyanbot.music.sre'})
                      )

    def template_timed_out(self, running_seconds):
        template = self.preprocess_template(u"""
            🚧 Выкатка {branch}/{revision} ({target}) не была завершена за {time} минут!
        """)
        return template.format(branch=self.Parameters.branch,
                               revision=self.Parameters.revision,
                               target=self.Parameters.target,
                               time=int(running_seconds / 60))

    def deploy_tests(self):

        self.enqueue_subtask(MusicDeployTests, issue=self.Parameters.issue)

    def report_task_done(self, token):
        """
        Report task done to Startrek release ticket

        :param token:  Startrek client token
        """
        st = StartrekHelper(token)

        template = self.preprocess_template(u"""
            Выкатка в {env} завершена!\n\n
            Ветка: {branch}
            Ревизия: {revision}
        """.format(env=self.Parameters.target,
                   branch=self.Parameters.branch,
                   revision=self.Parameters.revision))

        st.add_comment(self.Parameters.issue, template)

    def report_task_failed(self, token, running_seconds):
        """
        Report task failed to Startrek release ticket

        :param running_seconds: Seconds elapsed since deployment started
        :param token:  Startrek client token
        """
        st = StartrekHelper(token)

        template = self.preprocess_template(u"""
            Выкатка в {env} не была завершена за {time} минут!\n\n
            Ветка: {branch}
            Ревизия: {revision}
        """.format(env=self.Parameters.target,
                   branch=self.Parameters.branch,
                   revision=self.Parameters.revision,
                   time=int(running_seconds / 60)))

        st.add_comment(self.Parameters.issue, template)

    def mark_issue_testing_by_assessors(self, token):
        st = StartrekHelper(token)
        st.set_issue_status(self.Parameters.issue, 'testingByAssessors')

    def on_finish(self, *_):
        if not self.Parameters.set_juggler_downtime:
            return
        self._remove_downtime()

    def _set_downtime(self):
        from juggler_sdk import DowntimeSelector, JugglerError

        if not self.Parameters.set_juggler_downtime:
            return

        filters = [
            DowntimeSelector(
                service="junge",
                namespace="music.infra",
                host="music_production_statics"),
            DowntimeSelector(
                service="bazingaTasksMonitorAggregatePRODUCTION",
                namespace="music.backend",
                host="business-worker.music.yandex.net"),
            DowntimeSelector(
                service="junge_activity",
                namespace="music.infra",
                host="music_production_statics"),
        ]

        logging.info("Setting downtime for checks {}".format([d.fields for d in filters]))

        with self._juggler_api() as api:
            try:
                check = api.set_downtimes(filters,
                                          end_time=CONFIG.default_downtime_duration,
                                          description="Set by {}".format(type(self).__name__))
            except JugglerError:
                logging.error("Got a juggler error. Continue with the task so that startreck reporting does not break",
                              exc_info=True)
            else:
                self.Context.downtime_id = check.downtime_id

    def _remove_downtime(self):
        from juggler_sdk import JugglerError
        downtime_id = self.Context.downtime_id
        if not downtime_id:
            return

        with self._juggler_api() as api:
            try:
                removed_dts = api.remove_downtimes([downtime_id]).downtimes
            except JugglerError:
                logging.error("Got a juggler error. Continue with the task", exc_info=True)
            else:
                logging.info("Removed downtimes {}".format([d.downtime_id for d in removed_dts]))

    def _juggler_api(self):
        from juggler_sdk import JugglerApi

        token = sdk2.Vault.data(CONFIG.juggler_token)
        juggler_api = "http://juggler-api.search.yandex.net"
        return JugglerApi(juggler_api,
                          mark=type(self).__name__,
                          oauth_token=token)

    @staticmethod
    def disable_notification():
        hour = datetime.datetime.now().hour
        return hour < CONFIG.cron['hours']['from'] or hour > CONFIG.cron['hours']['to']

    def on_execute(self):
        dashboards = json.loads(self.Parameters.dashboards)
        token = sdk2.Vault.data(CONFIG.token)
        nanny = NannyClient(CONFIG.nanny_api_url, token)

        _nyan = partial(nyan, disable_notification=self.disable_notification())

        # get statuses
        all_ready = True
        for service in dashboards:
            for env in dashboards[service]:
                taskgroup = dashboards[service][env]['taskgroup']
                is_ready = (nanny.get_taskgroup_status(taskgroup)['status'] == 'DONE')
                dashboards[service][env]['is_ready'] = is_ready
                if not is_ready:
                    all_ready = False

        # check timeout
        running_seconds = (datetime.datetime.now(self.created.tzinfo) - self.created).total_seconds()
        if (self.Parameters.target == 'testing' and running_seconds > 3600) or \
                running_seconds > 12600:
            _nyan(self.template_timed_out(running_seconds))
            self.report_task_failed(token, running_seconds)
            if self.Parameters.target == 'testing':
                self.deploy_tests()

            return

        with self.memoize_stage.set_downtime:
            if self.Parameters.target in ['stable',
                                          'hotfix',
                                          'rollback_prod',
                                          'rollback_stable']:
                self._set_downtime()

        # create / update the message
        if self.Parameters.target == 'testing':
            message = self.template_testing(self.dashboards_to_deployments(dashboards, 'test'),
                                            self.dashboards_to_deployments(dashboards, 'qa'))
        elif self.Parameters.target == 'stable':
            message = self.template_stable(self.dashboards_to_deployments(dashboards, 'stable'),
                                           self.dashboards_to_deployments(dashboards, 'prestable'))
        elif self.Parameters.target == 'prestable':
            message = self.template_prestable(self.dashboards_to_deployments(dashboards, 'prestable'))
        elif self.Parameters.target == 'hotfix':
            message = self.template_hotfix(self.dashboards_to_deployments(dashboards, 'stable'),
                                           self.dashboards_to_deployments(dashboards, 'prestable'))
        elif self.Parameters.target.startswith('rollback'):

            if self.Parameters.target.endswith('prod'):
                message = self.template_rollback(self.dashboards_to_deployments(dashboards, 'stable'),
                                                 self.dashboards_to_deployments(dashboards, 'prestable'))

            elif self.Parameters.target.endswith('prestable'):
                message = self.template_rollback(prestable=self.dashboards_to_deployments(dashboards, 'prestable'))

            elif self.Parameters.target.endswith('stable'):
                message = self.template_rollback(stable=self.dashboards_to_deployments(dashboards, 'stable'))

            else:
                raise errors.TaskFailure("Unknown target {}".format(self.Parameters.target))
        else:
            raise errors.TaskFailure("Unknown target {}".format(self.Parameters.target))

        message_id = None if self.Context.message_id is NotExists else self.Context.message_id
        self.Context.message_id = _nyan(message, message_id)

        # check if ready
        if all_ready:
            if self.Parameters.target == 'testing':
                self.deploy_tests()
                # do not mark testing by assessors because they are an expensive resource
                # self.mark_issue_testing_by_assessors(token)
            _nyan(self.template_done(running_seconds))
            if self.Parameters.target == 'stable':
                self.grafana_event_done()

            self.report_task_done(token)
            return

        raise sdk2.WaitTime(300)
