import json
import logging
import os
import re
from copy import copy
from datetime import datetime, timedelta

import pymongo
import requests

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.nanny import const
from sandbox.projects.music.deployment.UpdateTestResources.helpers.MongoDumpUploader import MongoDumpUploader
from sandbox.projects.music.deployment.UpdateTestResources.helpers.StaticsOffheapArtifactsUploader \
    import StaticsOffheapArtifactsUploader
from sandbox.projects.music.deployment.UpdateTestResources.helpers.TestApiStaticsArtifactsUploader \
    import TestApiStaticsArtifactsUploader
from sandbox.projects.music.deployment.helpers.Config import CONFIG
from sandbox.projects.music.deployment.helpers.MusicBaseTask import MusicBaseTask
from sandbox.projects.music.environments import MongoUtilsEnvironment
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.svn import Arcadia, SvnError
from sandbox.sdk2 import yav


class HostNotFoundException(Exception):
    pass


class UpdateTestResources(nanny.ReleaseToNannyTask2, MusicBaseTask):
    class Requirements(sdk2.Task.Requirements):
        environments = (MongoUtilsEnvironment(), PipEnvironment("zstandard", version="0.12.0"))
        client_tags = Tag.LINUX_XENIAL
        disk_space = 40 * 1024

    class Parameters(sdk2.task.Parameters):
        description = "Update sandbox resources in tests."
        use_local_db = sdk2.parameters.Bool('Use ydb on localhost (for task development)', required=True, default=False)

    def on_execute(self):
        env = {}
        if not self.Parameters.use_local_db:
            env = self._get_env_secrets()

        yc_mongo_dump_resource = self._create_mongo_dump_resource(env)

        test_api_resource, test_api_prefix = self._create_test_api_resource(env)

        mongo_host = self._get_qa_mongo_host()
        basecaches_resource = self._create_basecaches_dump_resource(mongo_host, env)

        test_big_blob_prefix = self._get_fresh_prefix(mongo_host, env)
        test_big_blob_resource = self._create_testbigblob_resource(test_big_blob_prefix, mongo_host, env)

        self._commit_everything(basecaches_resource,
                                test_big_blob_resource,
                                test_big_blob_prefix,
                                test_api_resource.id,
                                test_api_prefix,
                                yc_mongo_dump_resource)

        self.set_info('Sending release for {} to Nanny'.format(test_api_resource.type.name))
        self._release_to_nanny(test_api_resource)
        self.set_info('All done')

    def _release_to_nanny(self, resource):
        resources = [{
            'id': str(resource.id),
            'type': resource.type.name,
            'description': resource.description,
            'skynetId': resource.skynet_id,
            'httpUrl': resource.http_proxy or '',
            'arch': resource.arch or '',
            'fileMd5': resource.md5 or '',
            'releasers': [self.author],
        }]
        nannyReleaseInfo = nanny.ReleaseToNannyTask2.get_nanny_release_info(self, dict(
            releaser=self.author,
            release_status=ctt.ReleaseStatus.TESTING,
            release_subject="Statics for music web tests",
        ))
        nannyReleaseInfo["spec"]["sandboxRelease"]["resources"] = resources
        nannyReleaseInfo["spec"]["sandboxRelease"]["resourceTypes"] = [resources[0]["type"]]

        logging.info('Release payload: %s', json.dumps(nannyReleaseInfo, indent=4))
        result = self.nanny_client.create_release2(nannyReleaseInfo)['value']
        logging.info('Release of task %s has been sent to Nanny', self.id)
        release_id = result['id']
        release_link = const.RELEASE_REQUEST_TEMPLATE.format(
            nanny_api_url=const.NANNY_API_URL,
            release_request_id=release_id,
        )
        self.set_info(
            'Nanny release request <a href="{}">{}</a> was created.'.format(release_link, release_id),
            do_escape=False
        )

    def _commit_everything(self,
                           basecaches_resource,
                           test_big_blob_resource,
                           test_big_blob_prefix,
                           test_api_resource,
                           test_api_prefix,
                           yc_mongo_dump_resource):
        self.set_info('Running stage: commit everything')
        checkout_path = str(self.path('music_repo'))
        trunk_url = Arcadia.trunk_url("music")
        Arcadia.checkout(trunk_url, checkout_path)

        change_list = CONFIG.get_update_test_resources_change_list(
            basecaches_resource,
            test_big_blob_resource,
            test_big_blob_prefix,
            test_api_resource,
            test_api_prefix,
            yc_mongo_dump_resource
        )
        self._update_config_files(checkout_path, change_list)

        url = self._commit_changes(checkout_path)
        self.set_info('Created PR: <a href="{url}">{url}</a>'.format(url=url), do_escape=False)

    def _create_basecaches_dump_resource(self, mongo_host, env):
        self.set_info('Running stage: dump basecaches from mongo')
        resource_id = MongoDumpUploader.execute(
            self,
            db_host=mongo_host,
            db_user=env["RECIPE_MONGO_MAIN_USER"],
            db_password=env["RECIPE_MONGO_MAIN_PASSWORD"],
            authdb="admin",
            dump_items=[("elliptics_metadata", ["basecaches"])],
            resource_ttl=30
        )
        return resource_id

    def _create_testbigblob_resource(self, prefix, mongo_host, env):
        self.set_info('Running stage: dump big statics from MDS')
        db_user = env["RECIPE_MONGO_MAIN_USER"]
        db_password = env["RECIPE_MONGO_MAIN_PASSWORD"]
        resource_id = StaticsOffheapArtifactsUploader.execute(self, mongo_host, db_user, db_password, prefix)
        return resource_id

    def _create_mongo_dump_resource(self, env):
        self.set_info('Running stage: dump yc mongo for statics tests')
        dump_items = [(db_name, None) for db_name in CONFIG.yc_mongo_databases_for_recipe_with_data_dump]
        resource_id = MongoDumpUploader.execute(
            self,
            db_host=env["RECIPE_MONGO_HOST"],
            db_user=env["RECIPE_MONGO_USER"],
            db_password=env["RECIPE_MONGO_PASSWORD"],
            authdb="admin",
            dump_items=dump_items,
            resource_ttl=30,
            db_port=env["RECIPE_MONGO_PORT"],
            prefix="yc",
        )
        return resource_id

    @staticmethod
    def _get_env_secrets():
        env = copy(CONFIG.yc_config_for_statics)
        for key in list(env.keys()):
            if key.endswith("PASSWORD"):
                secret_id, item = env[key].split("#")
                logging.info("Resolve item '%s' of the yav secret '%s'", item, secret_id)
                secret = yav.Secret(secret_id)  # last secret version
                password = secret.data()[item]
                env[key] = password
        return env

    def _create_test_api_resource(self, env):
        self.set_info('Running stage: dump small statics from YC')
        resource, prefix = TestApiStaticsArtifactsUploader.execute(self, env)
        return resource, prefix

    def _update_config_files(self, checkout_path, change_list):
        logging.info("Updating config files in Arcadia")

        for repo_file, changes in change_list.items():
            replaced = self._replace_matching_line(checkout_path, repo_file, changes)
            assert replaced == len(changes), "Exactly {} changes are expected, but {} done".format(
                len(changes), replaced)

    @staticmethod
    def _get_qa_mongo_host():
        r = requests.get("https://c.yandex-team.ru/api/groups2hosts/music-qa-mongo-main-mongos?format=json")
        r.raise_for_status()
        for mongo_host in r.json():
            return mongo_host["fqdn"]

        raise HostNotFoundException()

    @staticmethod
    def _get_fresh_prefix(mongo_host, env):
        """
        We can encounter a situation when basecaches dump is uploaded from prod to qa during statics generation.
        In that case there will be missing artifacts for latest prefix. To escape this situation method finds and
        returns (latest prefix - 1)
        """
        mongo_login = env["RECIPE_MONGO_MAIN_USER"]
        mongo_pass = env["RECIPE_MONGO_MAIN_PASSWORD"]
        uri = "mongodb://{login}:{password}@{host}".format(login=mongo_login, password=mongo_pass, host=mongo_host)
        query = {"_id": {"$regex": r"^statics/\d{4}-\d{2}-\d{2}T\d{2}-\d{2}-\d{2}"}}
        prefix_regex_pattern = r"(\d{4}-\d{2}-\d{2}T\d{2}-\d{2}-\d{2})"

        mongo = pymongo.MongoClient(uri)
        collection = mongo.elliptics_metadata.basecaches
        id_with_prefix = collection.find_one(query, sort=[("_id", pymongo.DESCENDING)])["_id"]
        prefix = re.search(prefix_regex_pattern, id_with_prefix).group(0)

        prefix_date_pattern = "%Y-%m-%dT%H-%M-%S"
        prev_prefix = (datetime.strptime(prefix, prefix_date_pattern) - timedelta(days=1)).strftime(prefix_date_pattern)
        logging.debug("Got prefix %s" % prev_prefix)
        return prev_prefix

    @staticmethod
    def _replace_matching_line(checkout_path, repo_file, changes):
        logging.debug("Trying to update file %s", repo_file)
        path = os.path.join(checkout_path, repo_file)
        with open(path, 'r') as f:
            lines = f.readlines()

        replaced_count = 0
        for change in changes:
            comment_regex = re.compile(change["comment_pattern"])
            for i in range(len(lines)):
                if comment_regex.match(lines[i]):
                    replaced_count += 1
                    # Change the line under the matched comment
                    lines[i + 1] = change["newline"]
                    break

        with open(path, 'w') as f:
            f.writelines(lines)
        return replaced_count

    @staticmethod
    def _commit_changes(checkout_path):
        svn_st = Arcadia.status(checkout_path)
        if svn_st:
            logging.info("Committing files:\n%s" % svn_st)
            commit_msg = "Update resource ids for music tests"
            try:
                Arcadia.commit(checkout_path, commit_msg, "zomb-sandbox-rw")
            except SvnError as err:
                match = re.search(r'(https://a.yandex-team.ru/review/\d+)', str(err))
                if match:
                    return match.group(1)
                else:
                    raise err
