import logging
import os
import shutil
import tarfile
import tempfile

from sandbox import sdk2
from sandbox.projects.music.resources import MongoCollectionDump
from sandbox.sdk2.helpers import subprocess


class MongoDumpUploader:
    """ Upload bzip2 compressed dump to Sandbox """

    def __init__(self):
        pass

    @classmethod
    def execute(cls, task, db_host, db_user, db_password, authdb, dump_items, resource_ttl=30, db_port="27017",
                prefix=""):
        compressed_dump_path = tempfile.NamedTemporaryFile().name
        uncompressed_dump_path = tempfile.NamedTemporaryFile().name

        for db_name, collections in dump_items:
            if collections:
                for db_collection in collections:
                    cmd = ["mongodump",
                           # https://dba.stackexchange.com/questions/215534/mongodump-unrecognized-field-snapshot
                           "--forceTableScan",
                           "--port", str(db_port),
                           "--host", db_host,
                           "--authenticationDatabase", authdb,
                           "-u", db_user,
                           "-p", db_password,
                           "-d", db_name,
                           "-c", db_collection,
                           "-o", uncompressed_dump_path]
                    cls._execute_cmd(cmd)
            else:
                cmd = ["mongodump",
                       "--forceTableScan",
                       "--port", str(db_port),
                       "--host", db_host,
                       "--authenticationDatabase", authdb,
                       "-u", db_user,
                       "-p", db_password,
                       "-d", db_name,
                       "-o", uncompressed_dump_path]
                cls._execute_cmd(cmd)
        cls._compress_dump(uncompressed_dump_path, compressed_dump_path)

        compressed_dump_size = os.path.getsize(compressed_dump_path) >> 20
        logging.info("Created compressed collection dump. Total size: {0:.2f} MB".format(compressed_dump_size))

        resource_id = cls._create_resource(task, prefix, compressed_dump_path, resource_ttl)
        return resource_id

    @staticmethod
    def _execute_cmd(cmd):
        process = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        out, err = process.communicate()
        exit_code = process.returncode
        logging.debug("Process out: '{}', error: '{}'".format(out, err))

        if exit_code != 0:
            logging.error(err)
            raise subprocess.CalledProcessError(exit_code, "mongodump", err)

    @staticmethod
    def _compress_dump(uncompressed_dump_path, compressed_dump_path):
        with tarfile.open(compressed_dump_path, "w:bz2") as tf:
            tf.add(uncompressed_dump_path, arcname="dump")

    @staticmethod
    def _create_resource(task, prefix, compressed_dump_path, resource_ttl):
        resource = MongoCollectionDump(task, "Mongo collection bzip2 compressed dump",
                                       os.path.join(prefix, "dump.tar.bz2"),
                                       ttl=resource_ttl)
        resource_data = sdk2.ResourceData(resource)
        dst_path = str(resource_data.path)
        dst_path_dir = os.path.dirname(dst_path)
        if not os.path.exists(dst_path_dir):
            os.makedirs(dst_path_dir)
        shutil.copy(compressed_dump_path, dst_path)
        logging.debug("Mongo resource path: %s" % resource_data.path)
        resource_data.ready()

        return resource.id
