import logging
import os
import shutil
import subprocess
try:
    from collections.abc import Callable
except ImportError:
    from collections import Callable
from datetime import datetime
from textwrap import dedent
from typing import List, cast

import requests
from sandbox.projects.music.deployment.helpers.Config import CONFIG
import sandbox.projects.common.arcadia.sdk as asdk
import sandbox.projects.music.resources as mres
import sandbox.sdk2.vcs.svn as svn
from sandbox import sdk2
from sandbox.sdk2 import Task, Resource

from .ArcadiaHelper import ArcadiaHelper
from .TaskHelper import TaskHelper

# these are for pyflakes tests to pass
assert Task
assert Resource
assert Callable


class BuildHelper(object):

    _BUILD_DIR = 'build'
    _OUT_DIR = 'out'
    _TARGETS = [
        'music/backend',
        'music/tools/subash',
        'music/tools/toolbox'
    ]

    def __init__(self,
                 patch,                         # type: str
                 jdks,                          # type: List[str]
                 set_info,                      # type: Callable[[str], None]
                 arcanum_token,                 # type: str
                 use_yt_cache=True,             # type: bool
                 move_files=False,              # type: bool
                 ):

        assert not set(jdks).difference(set(CONFIG._SUPPORTED_JDK_VERSIONS)), (
            "Only supported jdk versions {}".format(CONFIG._SUPPORTED_JDK_VERSIONS))
        self._jdks = jdks
        self._default_jdk = jdks[0]
        self._patch = patch
        self._set_info = set_info
        self._arcanum_token = arcanum_token
        self._use_yt_cache = use_yt_cache
        self._move_files = move_files

        self._branch = ''               # type: str
        self._revision = 0              # type: int

    def _patch_code(self,
                    path,               # type: str
                    ):

        if not self._patch:
            return

        patch = self._patch

        if patch.endswith('.zipatch'):
            self._set_info('Zipatching from {}'.format(patch))
            svn.Arcadia.apply_patch(path, 'zipatch:' + patch, '.')
            return

        filename = os.path.abspath('./own.arcadia.patch')
        self._set_info('Fetching {} into {}'.format(patch, filename))
        r = requests.get(patch, headers={'Authorization': 'OAuth {}'.format(self._arcanum_token)})
        r.raise_for_status()
        with open(filename, 'w') as fout:
            # temporary crutch, remove when https://a.yandex-team.ru/review/976255/files/1 gets merged in
            fout.write(str(r.text).replace('trunk/arcadia/', ''))

        logging.info(subprocess.check_output(['ls', '-l', filename]))
        self._set_info('Patching {}'.format(path))
        svn.Arcadia.apply_patch_file(path, filename, False)

    @staticmethod
    def _is_module(name):  # type: (str) -> bool
        return name.startswith('music') and name != 'music-common'

    @staticmethod
    def _kosher_copy_jars(in_dir,           # type: str
                          out_dir,          # type: str
                          modules,          # type: List[str]
                          move_files=False  # type: bool
                          ):

        sums = BuildHelper.checksum_path(in_dir, summer="md5sum")
        hashmap = {}
        for line in sums.splitlines():
            hashsum, filename = line.split()
            hashmap[filename] = hashsum

        out_data = os.path.join(out_dir, 'data')
        os.makedirs(out_data)
        for mod in modules:
            try:
                files = os.listdir(os.path.join(in_dir, mod, mod))
            except OSError as x:
                logging.error(x)
                continue
            os.makedirs(os.path.join(out_dir, mod))
            for file in files:
                source = os.path.join(in_dir, mod, mod, file)
                _hash = hashmap[source]
                target = os.path.join(out_data, _hash + "-" + file)
                symlink = os.path.join(out_dir, mod, file)
                if not os.path.exists(target):
                    if move_files:
                        logging.debug("Move %s -> %s", source, target)
                        shutil.move(os.path.join(in_dir, mod, mod, file), target)
                    else:
                        logging.debug("Copy %s -> %s", source, target)
                        shutil.copy(os.path.join(in_dir, mod, mod, file), target)
                logging.debug("Symlink %s -> %s", symlink, target)
                os.symlink("../data/" + os.path.basename(target), symlink)

    def _copy_jars(self, build_dir, base_out_dir):
        out_dir = os.path.join(base_out_dir, 'lib', 'jars')
        os.makedirs(out_dir)

        in_dir = os.path.join(build_dir, self._TARGETS[0])
        modules = []       # List[str]
        for name in os.listdir(in_dir):
            if self._is_module(name):
                modules.append(name)
        logging.info('Found these modules in {}: {}'.format(in_dir, modules))

        self._kosher_copy_jars(in_dir, out_dir, modules, move_files=self._move_files)

    def _build_code(self,
                    path,               # type: str
                    yt_cache_token,     # type: str
                    jdk,                # type: str
                    build_dir,          # type: str
                    ):
        self._set_info('Build started: path={}, jdk={}, build_dir={}'.format(path, jdk, build_dir))
        asdk.do_build(
            build_system=asdk.consts.YMAKE_BUILD_SYSTEM,
            source_root=path,
            targets=self._TARGETS,
            results_dir=build_dir,
            clear_build=False,
            def_flags={'JDK_VERSION': jdk},
            yt_store_params=asdk.YtStoreParams(
                True,
                yt_cache_token,
                threads=50
            ) if self._use_yt_cache else None,
            dump_graph_for_debug=False
        )
        self._set_info('Build finished')

    def _copy_bootstrap(self,
                        path,       # type: str
                        build_dir,  # type: str
                        out_dir,    # type: str
                        ):
        shutil.copytree('{}/music/bootstrap'.format(path),
                        os.path.join(out_dir, 'bootstrap'))
        shutil.copy('{}/{}/subash'.format(build_dir, self._TARGETS[1]),
                    '{}/bootstrap/common/usr/bin/subash'.format(out_dir))
        shutil.copy('{}/{}/toolbox'.format(build_dir, self._TARGETS[2]),
                    '{}/bootstrap/toolbox'.format(out_dir))

    def _generate_version_properties(self,
                                     url,       # type: str
                                     out_path,  # type: str
                                     ):
        date = datetime.now().strftime('%Y-%m-%d')
        time = datetime.now().strftime('%H:%M:%S')
        version_properties = dedent("""
                                    build.date={date}T{time}
                                    project.svn.url={url}
                                    project.svn.rev={revision}
                                    project.version={date}.{branch}.{revision}
                                    """.format(revision=self._revision,
                                               branch=self._branch,
                                               url=url,
                                               date=date, time=time))
        os.makedirs(out_path)
        with open(out_path + '/music-version.app.properties', 'w') as omvp:
            omvp.write(version_properties)

    @staticmethod
    def checksum_path(path, summer="sha256sum"):  # type: (str, str) -> str
        sums = subprocess.check_output(
            'find {} -type f -print0 | xargs -P8 -n50 -0 {}'.format(path, summer),
            shell=True
        )
        if isinstance(sums, bytes):
            sums = sums.decode('utf8')
        return sums

    def _checksum(self):
        self._set_info('Checksumming result')

        sums = self.checksum_path(self._OUT_DIR)
        sums = sums.replace('  out/bootstrap', '  bootstrap/bootstrap')
        sums = sums.replace('  out/lib', '  music/lib')
        open('{}/checksums'.format(self._OUT_DIR), 'w').write(sums)

    def _package_single_resource(self,
                                 task,              # type: Task
                                 cls,               # type: Resource
                                 description,       # type: str
                                 path,              # type: str
                                 add_attrs=False,   # type: bool
                                 ):
        resource = cast(Callable, cls)(
            task,
            description='Music JARs {}for revision {}/{}'.format(description,
                                                                 self._branch,
                                                                 self._revision),
            path='{}/{}'.format(self._OUT_DIR, path),
            arch=None,
            ttl=7 if self._patch else 30
        )
        if add_attrs:
            resource.branch = str(self._branch)
            resource.revision = int(self._revision)
            if self._patch is not None:
                resource.patch = self._patch
        data = sdk2.ResourceData(resource)
        data.ready()

    def _package_resources(self,
                           task,  # type: Task
                           ):
        self._set_info('Packaging resources')
        self._package_single_resource(task, mres.MusicJarsBootstrapResource, 'bootstrap ', 'bootstrap')
        self._package_single_resource(task, mres.MusicJarsChecksumResource, 'checksum ', 'checksums')
        self._package_single_resource(task, mres.MusicJarsResource, '', 'lib', not self._patch)

    def _execute_path(self,
                      url,          # type: str
                      path,         # type: str
                      yt_token,     # type: str
                      jdk,          # type: str
                      build_dir,    # type: str
                      out_dir,      # type: str
                      ):
        """
        Does only part of the work that requires to keep the mounted-in arcadia
        """
        self._set_info('Execute for url={}, path={} build={}, out={}'.format(url, path, build_dir, out_dir))
        self._patch_code(path)
        self._build_code(path, yt_token, jdk, build_dir)
        self._copy_jars(build_dir, out_dir)
        self._generate_version_properties(url, out_dir + '/lib/jars/common')
        self._copy_bootstrap(path, build_dir, out_dir)

    def _execute_path_for_jdks(self,
                               url,          # type: str
                               path,         # type: str
                               yt_token,     # type: str
                               ):
        # this method is to be completely removed in favor of _execute_path after the production is jdk15, or not?
        for jdk in set(self._jdks):
            bd = self._BUILD_DIR + '/{}'.format(jdk)
            od = self._OUT_DIR
            move = False
            if jdk != self._default_jdk:
                od = od + '/{}'.format(jdk)
                move = True

            self._execute_path(url, path, yt_token, jdk, bd, od)

            dst = self._OUT_DIR + '/lib/{}'.format(jdk)
            if move:
                src = od + '/lib'
                logging.debug("Move '%s' -> '%s'", src, dst)
                shutil.move(src, dst)
                shutil.rmtree(od)
            else:
                src = "."
                symlink_target = os.path.dirname(dst) + "/" + src
                logging.debug("Symlink '%s' -> '%s'", dst, symlink_target)
                os.symlink(src, dst)

    @property
    def branch(self):
        return self._branch

    @property
    def revision(self):
        return self._revision

    def run(self,
            task,                   # type: Task
            url,                    # type: str
            yt_token,               # type: str
            ):
        """
        Runs the build and packages the resources.
        In the process sets up the branch and revision properties on self.
        """

        self._branch, revision = TaskHelper.extract_branch_and_revision(url, cast(bool, CONFIG.is_dev))
        if CONFIG.is_dev:
            self._set_info('Running in local arcadia path mode')
            self._branch = "local"
            self._revision = cast(int, revision)
            self._execute_path_for_jdks(url, url, yt_token)
        else:
            self._revision = revision if revision else ArcadiaHelper.get_latest_affected_revision(url)
            with asdk.mount_arc_path(url) as path:
                self._execute_path_for_jdks(url, path, yt_token)

        self._checksum()
        self._package_resources(task)
