# coding: utf-8
import logging
import time

import sandbox.common.types.misc as ctm
from sandbox import common


# Source: https://github.com/bottlepy/bottle/commit/fa7733e075da0d790d809aa3d2f53071897e6f76
class cached_property(object):
    def __init__(self, func):
        self.func = func

    def __get__(self, obj, cls):
        if obj is None:
            return self

        value = obj.__dict__[self.func.__name__] = self.func(obj)
        return value


class Config(object):
    arcadia_trunk = 'arcadia:/arc/trunk/arcadia'
    arcadia_branch = 'arcadia:/arc/branches/music/stable-{}/arcadia'
    arc_url = 'https://a.yandex-team.ru/arc/commit/{}'

    auth_duty_url = 'https://u.yandex-team.ru/api/v1/duties/onduty/bekend-muzyki'
    auth_duty_nonstrict = ('noobgam',)
    auth_build = {
        'logins': ('alexbool', 'paulus'),
        'include': ('svc_muz_development',),
        'exclude': ('svc_muz_testing',),  # SOX rule: testers cannot issue a build
    }
    auth_release = {
        'logins': ('alexbool', 'paulus'),
        'include': ('svc_muz_testing',),
        'exclude': ('svc_muz_development',),  # SOX rule: testers cannot issue a build
    }

    build_jars_task_type = 'MUSIC_BUILD_JARS'
    start_deployment_task_types = ('MUSIC_START_DEPLOYMENT', 'MUSIC_DEPLOYMENT_CRON')
    ci_project_id = 787
    ci_toolchain = 'default-linux-x86_64-relwithdebinfo'
    ci_max_revisions = 500
    components = ('web', 'export', 'worker', 'java-match')
    components_content = ('origs', 'loader')
    components_sox = ('web', 'export', 'worker')
    cron = {
        'morning_maintenance_hour': 7,
        'hours': {
            'from': 11,
            'to': 19,
        },
        'days': {
            'from': 0,  # monday
            'to': 4,  # friday
        }
    }
    list_prefix = u'    • '
    nanny_api_url = 'https://nanny.yandex-team.ru'
    recipes = {
        'testing': 'deploy-test',
        'qa': 'deploy-qa',
        'stable': 'deploy-stable',
        'prestable': 'deploy-prestable',
    }
    dashboard_recipe_overrides = {
        'music-web': {
            'deploy-stable': 'deploy-stable-noman'
        }
    }
    token = 'ROBOT_MUSIC_BACKEND_OAUTH_TOKEN'
    arcanum_token = 'ROBOT_MUSIC_BACKEND_ARCANUM_TOKEN'
    export_token = 'ROBOT_MUSIC_BACKEND_EXPORT_TOKEN'
    yt_token = 'ROBOT_MUSIC_BACKEND_YT_TOKEN'
    yt_prod_token = 'ROBOT_MUSIC_YT_YT_TOKEN'
    ydb_prod_token = 'ROBOT_MUSIC_YDB_YDB_TOKEN'
    juggler_token = 'ROBOT_MUSIC_ADMIN_JUGGLER_TOKEN'
    cookie = 'ROBOT_MUSIC_BACKEND_COOKIE'
    mongo_password = 'MUSIC_MONGO_PASS'
    mds_basic_token = 'MUSIC_MDS_BASIC_TOKEN'

    yc_mongo_databases_for_recipe_with_data_dump = ["music_user_data", "music_main"]
    yc_config_for_statics = {
        # https://yc.yandex-team.ru/folders/foogeagp74rahmukp21r/managed-mongodb/cluster/mdbdqulu7fchk4hg6lql
        "RECIPE_MONGO_HOST": (
            "man-79b0ehue734zb6p9.db.yandex.net,"
            "sas-48uzlnyo27qkah8q.db.yandex.net,"
            "vla-z0uhmdwl5y5kbtts.db.yandex.net"
        ),
        "RECIPE_MONGO_PORT": "27018",
        "RECIPE_MONGO_USER": "music",
        "RECIPE_MONGO_PASSWORD": "sec-01e98bx2bss9mpgn4178d48mnn#mongo_password",
        "RECIPE_MONGO_MAIN_USER": "music",
        "RECIPE_MONGO_MAIN_PASSWORD": "sec-01dx0vbyda03xta8vzf3gaanp1#mongo.password",
        "RECIPE_MONGO_SSL": "true",
        # https://yc.yandex-team.ru/folders/foogeagp74rahmukp21r/managed-mysql/cluster/mdbks8j7dv5ks2hcl36l
        "RECIPE_MYSQL_HOST": "sas-zp4ztqxso1it7nei.db.yandex.net,vla-735di66u68pimixr.db.yandex.net,man-11t5lguaan5xx973.db.yandex.net",
        "RECIPE_MYSQL_PORT": "3306",
        "RECIPE_MYSQL_USER": "music",
        "RECIPE_MYSQL_PASSWORD": "sec-01dx0swt30fapte82ctemncw7p#main.db.cloud.password",
        "RECIPE_MYSQL_SSL": "false",
        "MAIN_DB_IS_DYNAMIC_DATASOURCE": "true",
    }
    test_blob_src_path = "music/backend/music-worker/src/test-blob"
    test_blob_test_filter = "*MetaTagTopTest*"

    nanny_frontend_paths = ['/music/frontend/musfront/stable', '/music/frontend/musfront/qa']

    stepanar_tg_chat = 1380514

    checklist = [
        ('[qa] Проверить статус автотестов (пофиксить проблемы или оповестить дежурного при необходимости)'),
        '[qa] Проверить ченджлог (задачи и коммиты, попавшие в релиз) на корректность и протестированность',
        '[qa] Если нет блокирующих проблем — дать ОК на релиз',
        '[dev] Выкатить на prestable',
        '[dev] Проверить ошибки на prestable и убедиться, что новых ошибок не добавилось',
        '[dev] Выкатить в prod',
        '[qa] Проверить задачи, которые нужно протестировать в prod',
    ]

    _JDK11 = "11"
    _JDK15 = "15"
    _DEFAULT_JDK_VERSION = _JDK15
    _SUPPORTED_JDK_VERSIONS = (_JDK11, _JDK15)

    @cached_property
    def monitored_deployment_test_names(self):
        return [
            "MusicFacedeployTest",
            "MusicApiTest",
            "MusicQuasarSampleAppTest",
            "MusicExportTest",
            "MusicTankRegressionTest",
            "MusicFrontendVersionCompareTest"
        ]

    @cached_property
    def deployment_tests(self):
        # delayed loading to work around the resolve loop
        from sandbox.projects.music.deployment.MusicApiTest import MusicApiTest
        from sandbox.projects.music.deployment.MusicQuasarSampleAppTest import MusicQuasarSampleAppTest
        from sandbox.projects.music.deployment.MusicExportTest import MusicExportTest
        from sandbox.projects.music.deployment.MusicFacedeployTest import MusicFacedeployTest
        from sandbox.projects.music.deployment.MusicTankRegressionTest import MusicTankRegressionTest
        from sandbox.projects.music.deployment.MusicFrontendVersionCompareTest import MusicFrontendVersionCompareTest

        # Ordered test execution
        # Tasks in each  list are executed in parallel
        # Lists are executed one after another
        return [
            [
                MusicTankRegressionTest,
                MusicFacedeployTest,
                MusicExportTest,
            ],
            [
                MusicApiTest,
                MusicQuasarSampleAppTest,
                MusicFrontendVersionCompareTest,
            ]
        ]

    @cached_property
    def components_non_sox(self):
        return tuple(set(self.components) - set(self.components_sox))

    @cached_property
    def arcadia_trunk_music(self):
        return self.arcadia_trunk + '/music'

    def arcadia_trunk_revision(self, revision):
        return '{}@{}'.format(self.arcadia_trunk, revision)

    @property
    def arcadia_description(self):
        return 'arcadia:/arc/{trunk,branches/music/stable-123}/arcadia[@revision]'

    @staticmethod
    def markdown_startrek(key):
        return u'[{0}](https://st.yandex-team.ru/{0})'.format(key)

    def arc_commit_url(self, revision):
        return self.arc_url.format(revision)

    @staticmethod
    def deployment_url(dashboard, deployment):
        url = u'https://nanny.yandex-team.ru/ui/#/services/dashboards/catalog/{0}/deployments/catalog/{1}/'
        return url.format(dashboard, deployment)

    def markdown_deployment(self, dashboard, deployment):
        return u'[{}]('.format(dashboard) + self.deployment_url(dashboard, deployment) + u')'

    @cached_property
    def is_dev(self):
        installation = common.config.Registry().common.installation
        installation_is_local = installation == ctm.Installation.LOCAL
        logging.error("Is dev=%s, installation=%s", installation_is_local, installation)
        return installation_is_local

    @property
    def nyan_url(self):
        return 'https://api.nbt.media.yandex.net/v2'

    @cached_property
    def dashboards(self):
        return tuple('music-' + x for x in self.components)

    @staticmethod
    def wikiformat_sandbox(task_id):
        return '((https://sandbox.yandex-team.ru/task/{0} {0}))'.format(task_id)

    @staticmethod
    def issue_summary(revision, branch):
        return u'Релиз бэкенда {} ({})'.format(revision, branch)

    @property
    def default_downtime_duration(self):
        downtime = 60 * 60 * 2  # two hours
        return time.time() + downtime

    @staticmethod
    def get_update_test_resources_change_list(
        basecaches_resource,
        test_big_blob_resource,
        test_big_blob_prefix,
        test_api_resource,
        test_api_prefix,
        yc_mongo_dump_resource
    ):
        change_list = {
            "recipes/mongodb/recipe-with-blobs.inc": [{
                "comment_pattern": r".*MARKER for UPDATE_TEST_RESOURCES task in Sandbox.*",
                "newline": "DATA(sbr://{})\n".format(basecaches_resource)}],
            "backend/common/src/main/resources/application-default.properties": [{
                "comment_pattern": r".*MARKER for UPDATE_TEST_RESOURCES task in Sandbox.*",
                "newline": "test.big.blob.prefix={}\n".format(test_big_blob_prefix)}],
            "backend/music-worker/src/test-big-blob/ya.make": [{
                "comment_pattern": r".*MARKER for UPDATE_TEST_RESOURCES task in Sandbox.*",
                "newline": "    sbr://{}\n".format(test_big_blob_resource)}],
            "backend/music-web/src/test-api/ya.make": [
                {"comment_pattern": r".*MARKER_A for UPDATE_TEST_RESOURCES task in Sandbox.*",
                 "newline": "ENV(MUSIC_TEST_API_BLOB_PREFIX={})\n".format(test_api_prefix)},
                {"comment_pattern": r".*MARKER_B for UPDATE_TEST_RESOURCES task in Sandbox.*",
                 "newline": "DATA(sbr://{})\n".format(test_api_resource)},
            ],
            "backend/music-common/web-test-api-local/ya.make": [
                {"comment_pattern": r".*MARKER_A for UPDATE_TEST_RESOURCES task in Sandbox.*",
                 "newline": "ENV(MUSIC_TEST_API_BLOB_PREFIX={})\n".format(test_api_prefix)},
                {"comment_pattern": r".*MARKER_B for UPDATE_TEST_RESOURCES task in Sandbox.*",
                 "newline": "DATA(sbr://{})\n".format(test_api_resource)},
            ],
            "backend/common/src/main/resources/application-development.properties": [
                {"comment_pattern": r".*MARKER_A for UPDATE_TEST_RESOURCES task in Sandbox.*",
                 "newline": "localweb.statics.downloadable.prefix={}\n".format(test_api_prefix)},
                {"comment_pattern": r".*MARKER_B for UPDATE_TEST_RESOURCES task in Sandbox.*",
                 "newline": "localweb.statics.downloadable.sandbox.resource.id={}\n".format(test_api_resource)},
            ],
            "recipes/mongodb/recipe-with-data.inc": [{
                "comment_pattern": r".*MARKER for UPDATE_TEST_RESOURCES task in Sandbox.*",
                "newline": "DATA(sbr://{})\n".format(yc_mongo_dump_resource)}],
        }
        return change_list


CONFIG = Config()
