import logging
import time
from threading import Thread

from sandbox.projects.common.nanny.client import NannyClient, NannyApiException
from sandbox.projects.music.deployment.helpers.Config import CONFIG


class NannyServiceUpdateThread(Thread):

    def __init__(self, nanny, service, task_id, branch_and_revision, build_jars_task_type):
        super(NannyServiceUpdateThread, self).__init__()

        self._nanny = nanny
        self._service = service
        self._task_id = task_id
        self._branch_and_revision = branch_and_revision
        self._build_jars_task_type = build_jars_task_type

        self.exception = None

    def run(self):
        """ Create a service snapshot with the corresponding version and return it's id """
        logging.info('Switching service %s to %s (task %s)', self._service, self._branch_and_revision, self._task_id)

        comment = 'Update to {} (task id {})'.format(self._branch_and_revision, self._task_id)
        for i in range(5):
            try:
                self._nanny.update_service_sandbox_file(self._service,
                                                        self._build_jars_task_type,
                                                        str(self._task_id),
                                                        comment=comment)
                self.exception = None
                break
            except NannyApiException as x:
                self.exception = x
                time.sleep(i + 1)  # i goes from zero
                logging.warning('Got error: %s', x)


class NannyServiceRollbackThread(Thread):
    def __init__(self, nanny, service):
        super(NannyServiceRollbackThread, self).__init__()

        self._nanny = nanny
        self.service = service

        self.exception = None
        self.rollback_snapshot_id = None
        self.music_build_jars_id = None

    def get_music_jars_revision(self, snapshot_id):
        sb_file = self._nanny.get_history_runtime_attrs(snapshot_id)["content"]["resources"]["sandbox_files"]
        for f in sb_file:
            if f['task_type'] == 'MUSIC_BUILD_JARS':
                return f['task_id']

    def run(self):
        try:
            snapshot_id = self._nanny.get_service(self.service)["current_state"]["content"][
                "rollback_snapshot"]["snapshot_id"]
            self.rollback_snapshot_id = snapshot_id
        except Exception as ex:
            self.exception = ex
            return

        comment = "Rolling back '{}' to '{}'".format(self.service, snapshot_id)

        logging.info("Rolling back service {} to snapshot_id '{}'".format(self.service, snapshot_id))

        for i in range(5):
            try:
                self._nanny.set_snapshot_state(self.service,
                                               snapshot_id,
                                               'PREPARED',
                                               comment=comment,
                                               set_as_current=True)
                self.music_build_jars_id = self.get_music_jars_revision(snapshot_id)
            except NannyApiException as x:
                self.exception = x
                time.sleep(i + 1)  # i goes from zero
                logging.warning('Got error: %s', x)
            except Exception as ex:
                self.exception = ex
                return


class DashboardUpdater(object):
    def __init__(self, token, build_jars_task_type=CONFIG.build_jars_task_type):
        self._nanny = NannyClient(CONFIG.nanny_api_url, token)
        self._build_jars_task_type = build_jars_task_type

    def set_and_update(self, author, dashboard, recipe_names, task_id, branch_and_revision):
        if dashboard in CONFIG.dashboard_recipe_overrides:
            recipe_names = [CONFIG.dashboard_recipe_overrides.get(recipe, recipe) for recipe in recipe_names]

        self.set_nanny_services(author, dashboard, recipe_names, task_id, branch_and_revision)
        return self.run_dashboard(dashboard, recipe_names)

    def set_nanny_services(self, author, dashboard, recipe_names, task_id, branch_and_revision):
        """ Sets the required versions in nanny services """

        if dashboard in CONFIG.dashboard_recipe_overrides:
            recipe_names = [CONFIG.dashboard_recipe_overrides.get(recipe, recipe) for recipe in recipe_names]
        logging.info('Processing services in dashboard %s switch to %s (task %s)',
                     dashboard, branch_and_revision, task_id)

        self.reject_old_deployment(author, dashboard, recipe_names)

        for recipe_name in recipe_names:
            services = self.recipe_to_services(dashboard, recipe_name)
            if not services:
                continue

            service_threads = [
                NannyServiceUpdateThread(self._nanny,
                                         service,
                                         task_id,
                                         branch_and_revision,
                                         self._build_jars_task_type) for service in services]

            for thread in service_threads:
                thread.start()

            one_of_exceptions = None
            for thread in service_threads:
                thread.join()
                if thread.exception:
                    one_of_exceptions = thread.exception

            if one_of_exceptions:
                raise one_of_exceptions

            logging.info('Dashboard %s services set', dashboard)

    def reject_old_deployment(self, author, dashboard, recipe_names):
        if dashboard in CONFIG.dashboard_recipe_overrides:
            recipe_names = [CONFIG.dashboard_recipe_overrides[dashboard].get(recipe, recipe) for recipe in recipe_names]

        for deployment in self._nanny.get_dashboard_taskgroups(dashboard):
            if deployment['author'] == author and deployment['meta']['dashboard_deployment']['recipe_id'] in recipe_names:
                logging.info('Rejecting an old deployment {}'.format(deployment['id']))
                self._nanny.set_taskgroup_status(deployment['id'])

    def set_rollback_revision(self, author, dashboard, recipe_names, task_id):
        """" Set a rollback snapshot id as current """

        logging.info('Processing services rollback in dashboard %s (task %s)',
                     dashboard, task_id)
        if dashboard in CONFIG.dashboard_recipe_overrides:
            recipe_names = [CONFIG.dashboard_recipe_overrides[dashboard].get(recipe, recipe) for recipe in recipe_names]

        self.reject_old_deployment(author, dashboard, recipe_names)

        services = []
        for recipe_name in recipe_names:
            services_from_recipe = self.recipe_to_services(dashboard, recipe_name)
            # different services exits in different recipes. Do not skip
            if services_from_recipe:
                services += services_from_recipe

        if not services:
            logging.warn('No services found for dashboard %s with recipes %s', dashboard, recipe_names)
            return {}

        service_threads = [
            NannyServiceRollbackThread(self._nanny, service)
            for service in services
        ]

        for thread in service_threads:
            thread.start()

        svc_to_sandbox_task = {}
        one_of_exceptions = None
        for thread in service_threads:
            thread.join()
            if thread.exception:
                one_of_exceptions = thread.exception
            svc_to_sandbox_task[thread.service] = thread.music_build_jars_id

        if one_of_exceptions:
            raise one_of_exceptions

        logging.info('Dashboard %s services set', dashboard)
        return svc_to_sandbox_task

    def recipe_to_services(self, dashboard, recipe_name):
        try:
            recipe = self._nanny.get_dashboard_recipe(dashboard, recipe_name)
        except IndexError:
            logging.info('Recipe {} is missing in dashboard {}'.format(recipe_name, dashboard))
            return None
        services = set([task['data']['params']['service_id']
                        for task in recipe['content']['tasks']
                        if task['data']['name'] == 'set_snapshot_target_state'])
        logging.info("Services {} from recipe {}".format(services, recipe_name))
        return services

    def run_dashboard(self, dashboard, recipe_names):
        """ Runs the required dashboard """
        logging.info('Starting dashboard %s', dashboard)
        if dashboard in CONFIG.dashboard_recipe_overrides:
            recipe_names = [CONFIG.dashboard_recipe_overrides[dashboard].get(recipe, recipe) for recipe in recipe_names]

        result = {}
        for recipe_name in recipe_names:
            try:
                dash = self._nanny.deploy_dashboard(dashboard, recipe_name, use_latest_snapshot=True)
                result[recipe_name] = {
                    'taskgroup': dash['taskgroup_id'],
                    'deployment': dash['_id'],
                }
            except IndexError:
                logging.info('Recipe %s is missing from dashboard %s', recipe_name, dashboard)

        logging.info('Dashboard %s started!', dashboard)
        return result
