# coding: utf-8

import logging
from collections import defaultdict
from datetime import datetime

import requests

import sandbox.common.types.resource as ctr
import sandbox.sdk2 as sdk2
from sandbox.projects.music.resources import MusicJarsResource
from sandbox.projects.release_machine.helpers.staff_helper import StaffApi
from .AbcHelper import AbcHelper
from .ArcadiaHelper import ArcadiaHelper
from .Config import CONFIG
from .Nyan import nyan
from .StartrekHelper import StartrekHelper
from .TaskHelper import TaskHelper


class Deployment(object):

    def __init__(self, abc_token=None):
        self._token = sdk2.Vault.data(CONFIG.token)

        self._staff = StaffApi(self._token)
        self._startrek = None
        self._arcadia = None
        self._abc_token = abc_token

    @property
    def startrek(self):
        if not self._startrek:
            self._startrek = StartrekHelper(self._token)
        return self._startrek

    @property
    def arcadia(self):
        if not self._arcadia:
            self._arcadia = ArcadiaHelper(self.startrek)
        return self._arcadia

    @property
    def staff(self):
        return self._staff

    @staticmethod
    def get_ci_url(revision, branch):
        from urllib import urlencode
        params = {
            'project_id': CONFIG.ci_project_id,
            'branch': branch,
            'revision': revision,
            'toolchain': CONFIG.ci_toolchain,
            'max_revisions': CONFIG.ci_max_revisions,
        }
        return 'https://ci.yandex-team.ru/api/v1.0/timeline/?' + urlencode(params)

    @staticmethod
    def find_build_task(revision, branch, pr_id=None, patch=None):
        task_attributes = {
            "revision": revision,
            "branch": branch,
        }
        if pr_id is not None:
            task_attributes["pr_id"] = pr_id
        if patch is not None:
            task_attributes["patch"] = patch
        resource = sdk2.Resource.find(
            type=MusicJarsResource,
            status=ctr.State.READY,
            attrs=task_attributes
        ).first()
        return resource.task if resource else None

    @staticmethod
    def get_revision_fail_count(revision, branch):
        """ Determine revision fail count. Does not raise errors but returns None on errors """
        url = Deployment.get_ci_url(revision, branch)
        logging.info('Checking revision {}/{} fail count from {}'.format(branch, revision, url))
        result = requests.get(url)
        try:
            result.raise_for_status()
            return int(result.json()[-1][u'fails'])
        except requests.HTTPError:
            return None
        except IndexError:
            return None

    @staticmethod
    def get_trunk_fail_count():
        url = 'https://ci.yandex-team.ru/api/v1.0/tests?new_stats=1&limit=1&project_id=787&offset=0'
        result = requests.get(url)
        try:
            result.raise_for_status()
            return int(result.json()['tests_stats']['FAILED'])
        except requests.HTTPError:
            return 0

    def find_latest_affected_revision_with_ci(self, check_greenness):
        revisions = self.arcadia.get_latest_affected_revisions(CONFIG.arcadia_trunk_music)
        for revision in revisions:
            if not check_greenness:
                return revision
            if self.get_revision_fail_count(revision, 'trunk') == 0:
                return revision
        return None

    def create_new_release_if_needed(self, open_issues, new_release_task, check_greenness):
        if open_issues:
            logging.info('Skipping a new release creation because there are open pending releases')
            return

        if self.startrek.find_nonclosed_issues():
            logging.info('Skipping a new release creation because there are nonclosed tickets')
            return

        revision = self.find_latest_affected_revision_with_ci(check_greenness)
        if not revision:
            return

        logging.info('Enqueing a new release from revision {}'.format(revision))
        new_release_task(url=CONFIG.arcadia_trunk_revision(revision)).enqueue()

    @staticmethod
    def inform_users_about_sox_violations(open_issues):
        if not open_issues:
            return

        text = u'🧦 Внимание, есть необработанные тикеты SOX!\n'
        for issue in open_issues:
            text += u'  · [{}](https://st.yandex-team.ru/{})\n'.format(issue.key, issue.key)
        nyan(text.strip())

    def inform_users_about_missing_startrek_fields(self, open_issues):
        import startrek_client.exceptions

        login_to_issues_documentation = defaultdict(list)
        for issue in open_issues:
            links = list(x.object for x in issue.links)
            for link in links:
                try:
                    if link.needDocumentation is None and link.assignee:
                        login_to_issues_documentation[link.assignee.id].append(link.key)
                except startrek_client.exceptions.StartrekError as err:
                    logging.warning('Got an ST error accessing %s: %s', link, err)

        all_logins = set(login_to_issues_documentation.keys())
        if not all_logins:
            logging.info('All tickets have necessary fields set')
            return

        need_doc_result = ''
        for login in sorted(all_logins):
            telegram_accounts = [x for x in self._staff.get_persons('persons', {
                'login': login,
                '_one': 1,
                '_fields': 'accounts',
            })['accounts'] if x['type'] == 'telegram']
            telegram_call = '@' + telegram_accounts[0]['value'] if telegram_accounts else login

            if login in login_to_issues_documentation:
                issues = login_to_issues_documentation[login]
                need_doc_result += '\n' + CONFIG.list_prefix + telegram_call.replace('_', '\_') + ': ' + \
                                   ', '.join(CONFIG.markdown_startrek(x) for x in issues)

        if need_doc_result:
            result = u'👋 Проставьте, пожалуйста, поля в следующих задачах:\n'
            if need_doc_result:
                result += u'\n_Need documentation:_\n' + need_doc_result + '\n'
            nyan(result.strip())

    def inform_users_about_non_green_trunk(self):
        fails = self.get_trunk_fail_count()
        logging.info('Trunk fail count: {}'.format(fails))
        if fails > 0:
            nyan(u'🆘 Ребята, ахтунг, транк [красный](https://ci.yandex-team.ru/project/787)!')

    def start_deploy_to_stable_if_required(self, open_issues, deploy_to_stable_task):
        logging.info('Initiating deployment to stable')
        if open_issues:
            logging.info('Skipping deploy to stable because there is an open release issue')
            return

        ready_to_deploy = self.startrek.find_ready_to_deploy_issues()
        if len(ready_to_deploy) != 1:
            logging.info('Ready to deploy issues count: {}'.format(len(ready_to_deploy)))
            return

        issue = ready_to_deploy[0]

        match = self.startrek.summary_revision_and_branch(summary=issue.summary)
        if not match:
            logging.error('Bad issue {} summary: {}'.format(issue.key, issue.summary))
            return

        build_task_id = int(issue.releaseNotes)
        tasks = list(sdk2.Task.find(id=build_task_id, children=True).limit(1))
        if not tasks:
            logging.error('Cannot find task {}'.format(build_task_id))
            return

        task = tasks[0]
        if str(task.type) != CONFIG.build_jars_task_type:
            logging.error('Task {} type is {}, not {}'.format(task.id, task.type, CONFIG.build_jars_task_type))
            return

        task = deploy_to_stable_task(task=task)
        issue.transitions['releasing'].execute()
        task.enqueue()

    def start_deploy_to_prestable_if_required(self, open_issues, deploy_to_prestable_task):
        logging.info('Initiating deployment to prestable')
        if open_issues:
            logging.info('Skipping deploy to prestable because there is an open release issue')
            return

        ready_to_deploy = self.startrek.find_ready_for_prestable_issues()
        if len(ready_to_deploy) != 1:
            logging.info('Ready to deploy to prestable issues count: {}'.format(len(ready_to_deploy)))
            return

        issue = ready_to_deploy[0]

        match = self.startrek.summary_revision_and_branch(summary=issue.summary)
        if not match:
            logging.error('Bad issue {} summary: {}'.format(issue.key, issue.summary))
            return

        build_task_id = int(issue.releaseNotes)
        tasks = list(sdk2.Task.find(id=build_task_id, children=True).limit(1))
        if not tasks:
            logging.error('Cannot find task {}'.format(build_task_id))
            return

        task = tasks[0]
        if str(task.type) != CONFIG.build_jars_task_type:
            logging.error('Task {} type is {}, not {}'.format(task.id, task.type, CONFIG.build_jars_task_type))
            return

        task = deploy_to_prestable_task(task=task)
        issue.transitions['prestable'].execute()
        task.enqueue()

    def update_open_issue(self, issue):
        import startrek_client.exceptions

        need_update = False
        for link in list(x.object for x in issue.links):
            try:
                if link.updatedAt > issue.updatedAt:
                    need_update = True
                    break
            except startrek_client.exceptions.StartrekError as err:
                logging.warning('ST Error: %s', err)

        match = self.startrek.summary_revision_and_branch(summary=issue.summary)
        if not match:
            logging.error('Bad issue {} summary: {}'.format(issue.key, issue.summary))
            return

        _, branch = match
        parsed_changelog, commits_without_task, first_rev, last_rev = self.arcadia.prepare_and_extract_latest_changelog(
            branch)

        th = TaskHelper()
        th.abc_helper = AbcHelper(self._abc_token) if self._abc_token else None
        text = self.arcadia.render_changelog(parsed_changelog=parsed_changelog,
                                             commits_without_tasks=commits_without_task,
                                             from_revision=first_rev,
                                             to_revision=last_rev,
                                             branch=branch,
                                             on_duty=th.backend_on_duty())

        if issue.description != text:
            need_update = True

        if not need_update:
            logging.info('No need to update the issue description {}'.format(issue.key))
            return

        logging.info('The issue description {} needs to be updated'.format(issue.key))

        issue.update(description=text, assignee=th.backend_on_duty())
        logging.info('Issue {} updated'.format(issue.key))

    @staticmethod
    def _is_valid_regular_time_period(now):
        hour = now.hour
        day = now.weekday()
        if hour < CONFIG.cron['hours']['from'] or hour > CONFIG.cron['hours']['to'] or \
                day < CONFIG.cron['days']['from'] or day > CONFIG.cron['days']['to']:
            logging.info('Skipping hourly cron due to the time bounds')
            return False
        return True

    def morning_maintenance(self, now):
        dow = now.weekday()
        if now.hour != CONFIG.cron['morning_maintenance_hour'] or \
                dow < CONFIG.cron['days']['from'] or dow > CONFIG.cron['days']['to']:
            logging.info('Skipping morning maintenance due to time bounds: hour={}, dow={}'.format(now.hour, dow))
            return

        issues = self.startrek.find_releasing_issues()
        logging.info('Releasing issues count: {}'.format(len(issues)))
        if len(issues) > 1:
            logging.info('Skipping closing issues as there are many open')

        for issue in issues:
            logging.info('Closing issue {}'.format(issue.key))
            issue.transitions['closed'].execute(resolution='fixed', comment=u'Автозакрытие выкаченной задачи')

    def cron_hourly(self, do_morning_maintenance, inform_startrek_fields, inform_red_trunk):
        now = datetime.now()

        if do_morning_maintenance:
            self.morning_maintenance(now)

        if not self._is_valid_regular_time_period(now):
            return

        # inform sox -- always(!)
        sox_issues = self.startrek.find_sox_checksum_validation_errors_issues()
        self.inform_users_about_sox_violations(sox_issues)

        if inform_startrek_fields:
            open_issues = self.startrek.find_open_and_assessors_issues()
            self.inform_users_about_missing_startrek_fields(open_issues)

        if inform_red_trunk:
            self.inform_users_about_non_green_trunk()

    def cron_tenminute(self, deploy_to_stable_task, new_release_task, deploy_to_prestable_task, check_greenness):
        open_issues = self.startrek.find_open_and_assessors_issues()
        for issue in open_issues:
            self.update_open_issue(issue)

        # the next ones must be checked regularly, but executed only in allotted time
        self.create_new_release_if_needed(open_issues, new_release_task, check_greenness)
        self.start_deploy_to_stable_if_required(open_issues, deploy_to_stable_task)
        self.start_deploy_to_prestable_if_required(open_issues, deploy_to_prestable_task)
