# coding: utf-8

import json
import logging
import re

import requests


def startrek_conflict_retry(count):
    """ Retries decorated method :count times. Requires self._Conflict to be set """

    def outer(fun):
        def inner(self, *args, **kwargs):
            for i in range(count):
                try:
                    return fun(self, *args, **kwargs)
                except self._Conflict:
                    logging.warning('Got a conflict updating task on try {}'.format(i + 1))
                    pass

        return inner

    return outer


class StartrekHelper(object):
    _st_client_arcadia_path = "arcadia:/arc/trunk/arcadia/library/python/startrek_python_client"

    _ticket_summary_re = re.compile(r' (\d+) \((\S+)\)')
    _hotfix_marker_re = re.compile(u'^Запущена выкатка ХОТФИКСА')

    def __init__(self, token, queue='MUSICRELEASES'):
        import startrek_client
        self._token = token
        self._queue = queue
        self._startrek = startrek_client.Startrek(useragent=self.__class__.__name__, token=token)
        self._startrek_cache = {}
        self._Conflict = startrek_client.exceptions.Conflict

    def get_issue(self, key):
        return self._startrek.issues[key]

    @startrek_conflict_retry(3)
    def update_issue(self, key, description):
        issue = self._startrek.issues[key]
        issue.update(description=description)
        return key

    @startrek_conflict_retry(3)
    def set_issue_release_notes(self, key, notes):
        issue = self._startrek.issues[key]
        issue.update(releaseNotes=notes)
        return key

    def create_issue(self, summary, description, checklist, assignee=None, qa_engineer=None):
        issue = self._startrek.issues.create(queue=self._queue,
                                             summary=summary,
                                             description=description,
                                             assignee=assignee,
                                             qaEngineer=qa_engineer)

        # ST client does not handle checklist items yet
        items = [{'text': text, 'checked': False} for text in checklist]
        requests.post('https://st-api.yandex-team.ru/v2/issues/{}/checklistItems'.format(issue.key),
                      data=json.dumps(items),
                      headers={
                          'Authorization': 'OAuth {}'.format(self._token),
                          'Content-type': 'application/json'
        })
        self.append_last_hotfix_comment_from_previous_issue_to_all_openned_issues(issue.key)
        return issue

    def find_issue_for_branch(self, branch):
        items = list(self._startrek.issues.find(query="Queue: {} AND Summary: \"({})\"".format(self._queue, branch)))
        return items[0] if items else None

    def find_issues_in_state(self, state):
        return list(self._startrek.issues.find(query="Queue: {} AND Status: {}".format(self._queue, state)))

    def find_nonclosed_issues(self):
        return list(self._startrek.issues.find(query="Queue: {} AND Resolution: empty()".format(self._queue)))

    def find_open_issues(self):
        return self.find_issues_in_state('open')

    def find_issues_by_filter_id(self, filter_id):
        return list(self._startrek.issues.find(filter_id=filter_id))

    def find_tested_issues(self):
        return self.find_issues_in_state('tested')

    def find_releasing_issues(self):
        return self.find_issues_in_state('releasing')

    def find_releasing_to_prestable_issues(self):
        return self.find_issues_in_state('prestable')

    def find_ready_to_deploy_issues(self):
        return self.find_issues_in_state('readyToDeploy')

    def find_ready_for_prestable_issues(self):
        return self.find_issues_in_state('readyForPrestable')

    def find_open_and_assessors_issues(self):
        return self.find_issues_in_state('open,testingByAssessors')

    def find_sox_checksum_validation_errors_issues(self):
        query = 'queue: MUSICBACKEND and Summary: "Code checksum validation errors" and Resolution: empty()'
        return list(self._startrek.issues.find(query))

    def find_next_issue(self, key):
        idx = key.split('-')[1]
        return self._startrek.issues['{}-{}'.format(self._queue, int(idx) + 1)]

    def find_prev_issue(self, key):
        idx = key.split('-')[1]
        return self._startrek.issues['{}-{}'.format(self._queue, int(idx) - 1)]

    def ticket_assignee(self, key):
        issue = self._startrek.issues[key]
        if issue.assignee:
            return issue.assignee.login

    @startrek_conflict_retry(2)
    def add_comment(self, key, comment, summonee=None):
        summonees = [summonee] if summonee else None
        issue = self._startrek.issues[key]
        issue.comments.create(text=comment, summonees=summonees)

    @startrek_conflict_retry(2)
    def append_quoted_comment(self, key, comment, description=u""):
        if not isinstance(comment, bytes):
            comment = comment.decode('utf8')
        if not isinstance(description, bytes):
            description = description.decode('utf8')
        quoted_text = u'{}\n<[{}]>'.format(description, comment)
        self.get_issue(key).comments.create(text=quoted_text)

    def find_last_comment(self, issue, pattern):
        comments = reversed(list(issue.comments.get_all()))
        for c in comments:
            if pattern.match(c.text):
                return c
        return None

    def append_last_hotfix_comment_from_previous_issue_to_all_openned_issues(self, key):
        try:
            prev_issue = self.find_prev_issue(key)
            last_hotfix_comment = self.find_last_comment(prev_issue, self._hotfix_marker_re)
            if last_hotfix_comment is not None:
                logging.info("Found hotfix comment from issue %s", prev_issue.key)

                # explicitly add comment for current task
                # find_nonclosed_issues can fail to find just created isssue
                self.append_quoted_comment(
                    key,
                    last_hotfix_comment.text,
                    u'В предыдущем релизе были хотфиксы'
                )

                opened_issues = self.find_nonclosed_issues()
                for oi in opened_issues:
                    if oi.key == key:
                        continue
                    if oi.key < key:
                        logging.debug("Skip obsolete release %s", oi.key)
                        continue
                    self.append_quoted_comment(
                        oi.key,
                        last_hotfix_comment.text,
                        u'В предыдущем релизе были хотфиксы'
                    )
            else:
                logging.info("Previous release %s has no hotfixes", prev_issue.key)
        except Exception:  # noqa
            # to be removed if all work fine
            logging.exception("Your code is broken, please fix it!")

    @startrek_conflict_retry(3)
    def set_issue_status(self, key, new_status):
        issue = self._startrek.issues[key]
        transitions = [x for x in issue.transitions.get_all() if x.to.key == new_status]
        if not transitions:
            raise RuntimeError('Could not find a transition to {} for issue {}'.format(new_status, key))
        transitions[0].execute()

    def get_info(self, key):
        if key not in self._startrek_cache:
            try:
                issue = self._startrek.issues[key]

                if issue.qaEngineer:
                    qa_engineer = issue.qaEngineer.id
                else:
                    qa_engineer = u''

                self._startrek_cache[key] = {
                    'qa_engineer': qa_engineer,
                    'status': issue.status.key,
                    'releaseNotes': issue.releaseNotes,
                    'components': [c.name for c in issue.components],
                }
            except Exception as x:
                logging.error('Got exception retrieving ticket %s: %s', key, x)
                self._startrek_cache[key] = {
                    'qa_engineer': '',
                    'status': '',
                    'releaseNotes': '',
                    'components': [],
                }

        return self._startrek_cache[key]

    def summary_revision_and_branch(self, summary):
        match = re.search(self._ticket_summary_re, summary)
        if match:
            revision = int(match.group(1))
            branch = match.group(2)
            return (revision, branch)
        return None

    def patch_release_issue_statuses_raw(self, key, possible_names, succeeding_name):
        """ Sets the issue releaseComment field to current status and deployed: Да if all are successful """
        issue = self._startrek.issues[key]

        data = {}
        for name in possible_names:
            data[name] = False

        # noinspection PyBroadException
        try:
            if issue.releaseComment:
                data = json.loads(issue.releaseComment)
        except:
            logging.error('Could not parse releaseComment')

        data[succeeding_name] = True

        overall_good = True
        for key in data:
            if not data[key]:
                overall_good = False
                break

        issue.update(releaseComment=json.dumps(data), deployed='Да' if overall_good else None)
