# coding: utf-8

import logging
import re
from typing import Optional, Tuple

import requests

import sandbox.common.types.task as ctt
import sandbox.sandboxsdk.environments as environments
import sandbox.sdk2 as sdk2
from sandbox.common import errors
from .AbcHelper import AbcHelper, MUSIC_BACKEND_SCHEDULE, MUSIC_BACKEND_SERVICE, MUSIC_QA_SCHEDULE
from .Config import CONFIG

STATUS_FINISH = [
    ctt.Status.Group.FINISH,
    ctt.Status.Group.BREAK,
]

STATUS_RUN = [
    ctt.Status.Group.EXECUTE,
    ctt.Status.Group.WAIT,
]


def subtaskable(do_raise):
    def real_subtaskable(func):
        def subtask_list(parent, func_args):
            subtasks = sdk2.Task.find(parent=parent).limit(25)
            subtask_dict = dict((str(task.type), task) for task in subtasks if task.status != ctt.Status.DRAFT)
            return [subtask_dict.get(arg.upper(), None) for arg in func_args]

        def wrapper(self, *args, **kwargs):
            func_name = func.__name__
            func_args = [x for x in func.func_code.co_varnames if x.startswith(('music',
                                                                                'rollback_commit',
                                                                                'create_branch'))]

            if func_args:
                with getattr(self.memoize_stage, func_name + '::wait')():
                    wait = [task for task in subtask_list(self, func_args) if task]
                    if wait:
                        raise sdk2.WaitTask(wait, STATUS_FINISH, wait_all=True)

            with getattr(self.memoize_stage, func_name + '::run')():
                extra_args = subtask_list(self, func_args) if func_args else []
                if do_raise:
                    for arg in extra_args:
                        if arg and arg.status not in ctt.Status.Group.SUCCEED:
                            raise errors.TaskFailure('Subtask {} (#{}) has failed'.format(arg.type, arg.id))
                args = args + tuple(extra_args)

                return func(self, *args, **kwargs)

        return wrapper

    return real_subtaskable


class TaskHelper(object):
    startrek_client_environment = environments.PipEnvironment(
        "startrek_client", version="2.5", custom_parameters=["requests==2.18.4"], use_wheel=False)
    juggler_client_environment = environments.PipEnvironment("juggler-sdk", version="0.6.7")
    pydns_library_environment = environments.PipEnvironment("pydns", version="2.3.6")
    mysql_library_environment = environments.PipEnvironment("MySQL-python", version="1.2.5")

    _url_re = re.compile(r'^arcadia:/arc/(?:trunk|branches/music/([^/]+))/arcadia(?:@(\d+))?$')
    abc_helper = None  # type: Optional[AbcHelper]

    def enqueue_subtask(self, task_class, **kwargs):
        """
        :param task_class: sdk2.Task
        :type self: sdk2.Task
        """
        task = task_class(self, notifications=self.Parameters.notifications, **kwargs)
        task.enqueue()
        return task

    def check_there_is_only_one_such_task(self):
        """
        :type self: sdk2.Task
        """
        tasks = list(sdk2.Task.find(task_type=self.type,
                                    status=STATUS_RUN,
                                    children=True).limit(2))
        if len(tasks) > 1:
            raise errors.TaskFailure('Found running {} tasks: {}'.format(self.type,
                                                                         ", ".join(str(t.id) for t in tasks)))

    @staticmethod
    def combine_branch_and_revision_to_url(branch, revision):
        return 'arcadia:/arc/{}/arcadia@{}'.format(branch, revision)

    @staticmethod
    def extract_branch_and_revision(url, is_dev=False):  # type: (str, bool) -> Tuple[str, Optional[int]]
        m = TaskHelper._url_re.match(url)
        if not m:
            if is_dev:
                logging.error("url=%s not match %s", url, TaskHelper._url_re)
                return 'trunk', 11111
            raise errors.TaskFailure('The URL is in incompatible format')
        branch, revision = m.groups()
        if not branch:
            branch = 'trunk'

        if not revision:
            return branch, None
        return branch, int(revision)

    def _report_old_usage(self):
        from sandbox.projects.music.deployment.helpers.Nyan import nyan
        msg = "Task '{}' is using u.y-t.ru for on duty detection".format(self.__class__.__name__)
        nyan(msg, chat_id=-1001270499102)

    def backend_on_duty(self):
        on_duty_login = None

        if self.abc_helper:
            on_duty = self.abc_helper.get_onduty(MUSIC_BACKEND_SERVICE, MUSIC_BACKEND_SCHEDULE)
            on_duty_login = on_duty[0]['person']['login']
        else:
            self._report_old_usage()
            for _ in range(3):
                try:
                    r = requests.get(CONFIG.auth_duty_url)
                    r.raise_for_status()
                    on_duty_login = r.json()['onDuty']
                    break
                except requests.RequestException as err:
                    logging.warning('Cannot determine person on duty: %s', err)

        return on_duty_login

    def qa_on_duty(self):
        try:
            if self.abc_helper:
                on_duty = self.abc_helper.get_onduty(MUSIC_BACKEND_SERVICE, MUSIC_QA_SCHEDULE)
                return on_duty[0]['person']['login']
        except:
            return None

    def is_on_duty(self, login):
        on_duty_login = self.backend_on_duty()
        return (on_duty_login is None) or (on_duty_login in CONFIG.auth_duty_nonstrict) or (login == on_duty_login)

    def check_authorization(self, login, token_key, auth_config, force_duty=False):
        from sandbox.projects.release_machine.helpers.staff_helper import StaffApi
        if CONFIG.is_dev:
            logging.info('Allowing from dev host')
            return
        token = sdk2.Vault.data(token_key)

        # fallback for logins
        if login in auth_config['logins']:
            logging.info('Allowing build access for {} (explicit login)'.format(login))
            return

        # check duty
        if force_duty and not self.is_on_duty(login):
            raise RuntimeError('This task can only be launched by the person on music backend duty')

        staff_api = StaffApi(token)
        staff_groups = set(staff_api.get_user_groups(login))

        # check exclusions
        for group in auth_config['exclude']:
            if group in staff_groups:
                raise RuntimeError('Author {} is in the excluded build group {}'.format(login, group))

        # check inclusions
        for group in auth_config['include']:
            if group in staff_groups:
                logging.info('Allowing build access for {} (group {})'.format(login, group))
                return

        raise errors.AuthorizationError('Author {} is not present in any of {} groups'
                                        .format(login, ", ".join(auth_config['include'])))
