# -*- coding: utf-8 -*-
import logging
import os
import shutil

import sandbox.common.types.task as ctt
import sandbox.common.types.notification as ctn

from sandbox import sdk2
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.WizardRuntimeBuild.ya_make import YaMake


BINARY_EXTENSIONS = ['bin', 'trie', 'data']


class MusicMagicData(sdk2.Resource):
    pass


class MusicMagicSandboxFile(sdk2.Resource):
    pass


class CommitMusicMagicData(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        music_magic_data_path = sdk2.parameters.String(
            'MusicMagic data path',
            default=Arcadia.trunk_url('search/wizard/data/fresh/music_magic')
        )

        music_magic_data = sdk2.parameters.Resource(
            'new MusicMagic data resource id',
            required=True,
            resource_type=MusicMagicData,
        )

        notifications = [sdk2.Notification(
            [ctt.Status.FAILURE, ctt.Status.Group.BREAK],
            ['epinkovskii'],
            ctn.Transport.EMAIL,
        )]

    @staticmethod
    def __drop_resources(resource_ids):
        logging.info('About to drop following resources: %s', resource_ids)
        for resource_id in resource_ids:
            logging.info('Droping %s', resource_id)
            channel.sandbox.drop_resource_attribute(resource_id, 'ttl')

    @staticmethod
    def __is_binary_or_large(file_):
        if os.path.splitext(file_.name)[1] in BINARY_EXTENSIONS:
            return True
        return file_.stat().st_size >= 1024 * 1024

    def __update_wizard_data(self, new_data_dir, dest_data_dir):
        logging.info('Updating target data')
        ymake_path = os.path.join(dest_data_dir, 'ya.make')
        ymake = YaMake.YaMake(ymake_path)

        for file_ in new_data_dir.iterdir():
            source = os.path.join(str(new_data_dir), file_.name)
            dest = os.path.join(dest_data_dir, file_.name)
            existed = os.path.exists(dest)
            sandboxed = ymake.get_sandboxed(dest)

            if self.__is_binary_or_large(file_):
                if existed:
                    logging.info('%s changed storage to sandbox', file_.name)
                    Arcadia.delete(dest)

                new_sandbox_resource = MusicMagicSandboxFile(
                    self,
                    '{} for new MusicMagic data'.format(file_.name),
                    file_.name,
                    ttl="inf",
                    owner=self.owner)
                resource_data = sdk2.ResourceData(new_sandbox_resource)
                with open(source, 'r') as f:
                    resource_data.path.write_bytes(f.read())
                resource_data.ready()
                self.Context.new_resources.append(new_sandbox_resource.id)

                logging.info('Changing sandbox resource for file %s from %d to %d',
                             file_.name, sandboxed.resource, new_sandbox_resource.id)
                ymake.update_sandbox_resource(file_.name,
                                              id=new_sandbox_resource.id,
                                              compression='FILE')
                if sandboxed.resource != new_sandbox_resource.id and sandboxed.resource:
                    self.Context.obsolete_resources.append(sandboxed.resource)
            else:
                shutil.copy(source, dest)
                if sandboxed.resource or not existed:
                    ymake.remove_sandbox_resource(file_.name)
                    Arcadia.add(dest, parents=True)

        with open(ymake_path, 'w') as f:
            ymake.dump(f)

    def __cleanup_old_resources(self):
        self.__drop_resources(self.Context.obsolete_resources)

    def __remove_new_resources(self):
        self.__drop_resources(self.Context.new_resources)

    def on_execute(self):
        self.Context.new_resources = []
        self.Context.obsolete_resources = []

        wizard_data_path = str(self.path('wizard_data'))
        new_music_magic_data_path = sdk2.ResourceData(self.Parameters.music_magic_data).path

        Arcadia.checkout(self.Parameters.music_magic_data_path, wizard_data_path)
        self.__update_wizard_data(new_music_magic_data_path, wizard_data_path)
        try:
            svn_st = Arcadia.status(wizard_data_path)
            if svn_st:
                logging.info('Commiting new data')
                Arcadia.commit(wizard_data_path, 'new MusicMagic data', 'zomb-sandbox-rw')
            else:
                logging.info("No diff in data")
        except Exception:
            logging.error('Error commiting to arcadia', exc_info=True)
            self.__remove_new_resources()
        self.__cleanup_old_resources()
