import os
import logging
import contextlib
import tempfile
import shutil
import tarfile

from sandbox.common.types.client import Tag
from sandbox import sdk2


@contextlib.contextmanager
def create_tmp_dir(*args, **kwargs):
    path = tempfile.mkdtemp(*args, **kwargs)
    yield path
    shutil.rmtree(path)


def make_archive_from_content(src, arc, mode):
    with tarfile.open(arc, mode) as tf:
        for f in os.listdir(src):
            p = os.path.join(src, f)
            tf.add(p, f)


class NannyScliBin(sdk2.Resource):
    any_arch = False
    executable = True
    releasable = False


class BackupNannyArchive(sdk2.Resource):
    any_arch = True
    executable = False
    releasable = False


class BackupNannyServices(sdk2.Task):

    SECRETS_OWNER = 'NANNY'
    BACKUP_TTL = 30
    ARCHIVE_NAME = 'nanny_backup'

    class Requirements(sdk2.Task.Requirements):
        client_tags = Tag.LINUX_PRECISE
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        scli_bin = sdk2.parameters.Resource('Program binary', resource_type=NannyScliBin)
        service_id = sdk2.parameters.String('Service ID')
        workers = sdk2.parameters.Integer('Workers count', default=5)

    def get_token(self):
        return sdk2.Vault.data(self.SECRETS_OWNER, 'nanny_robot_oauth_token')

    def get_scli_binary(self):
        r = self.Parameters.scli_bin
        if r:
            return r
        return sdk2.Resource[NannyScliBin].find(state='READY').order(-sdk2.Resource.id).first()

    def on_execute(self):
        scli = self.get_scli_binary()
        if not scli:
            logging.error("Can't find scli binary")
            return

        with create_tmp_dir() as tmp:
            p = os.path.join(tmp, self.ARCHIVE_NAME)
            os.mkdir(p)

            cmd = []
            cmd.append(str(sdk2.ResourceData(scli).path))
            cmd.extend(['-d', p])
            cmd.append('dump')
            s = self.Parameters.service_id
            if s:
                cmd.append(s)
            else:
                cmd.append('--all')
            cmd.extend(['-w', str(self.Parameters.workers)])

            env = {'NANNY_OAUTH_TOKEN': self.get_token()}

            with sdk2.helpers.ProcessLog(self, logger='scli') as pl:
                sdk2.helpers.subprocess.check_output(cmd,
                                                     stderr=pl.stderr,
                                                     env=env)

            arc_path = str(self.path('{}.tar.gz'.format(self.ARCHIVE_NAME)))
            make_archive_from_content(tmp, arc_path, 'w:gz')
            r = BackupNannyArchive(self, 'Nanny backup', arc_path, 'linux', ttl=30)
            sdk2.ResourceData(r).ready()
