# -*- coding: utf-8 -*-
import os
import tarfile

from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny
from sandbox.projects.nanny.common.build_helpers import git_clone_and_checkout
from sandbox.projects.nanny.common.gobabygo_parameters import GitRefIdParameter, GitRefShaParameter, ReleaseParameter
from sandbox.projects.resource_types import AbstractResource, nanny_releasers
from sandbox.sandboxsdk.environments import VirtualEnvironment

from sandbox.sandboxsdk.task import SandboxTask


class ALERT_SYNCER_CODE(AbstractResource):
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    releasers = nanny_releasers


class ALERT_SYNCER_VIRTUAL_ENV(AbstractResource):
    shard = False
    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    releasers = nanny_releasers


class BuildNannyAlertSyncer(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BUILD_NANNY_ALERT_SYNCER'
    code_path = 'alert_syncer'
    req_file_name = 'pip-requirements.txt'
    wheels_dir_name = 'wheels'
    tar_code_name = 'code.tar.gz'
    tar_venv_name = 'venv.tar.gz'
    GIT_URL = 'https://{username}:{password}@bitbucket.yandex-team.ru/scm/Nanny/nanny'
    input_parameters = [GitRefIdParameter, GitRefShaParameter, ReleaseParameter]

    def _get_clone_url(self):
        oauth_token = self.get_vault_data('GBG', 'nanny_robot_bb_oauth_token')
        return self.GIT_URL.format(username='x-oauth-token', password=oauth_token)

    def on_execute(self):
        ref_id = utils.get_or_default(self.ctx, GitRefIdParameter)
        ref_sha = utils.get_or_default(self.ctx, GitRefShaParameter)
        release = utils.get_or_default(self.ctx, ReleaseParameter)
        package_path = self.path('nanny')
        clone_url = self._get_clone_url()

        git_clone_and_checkout(clone_url, package_path, ref_sha or ref_id)

        if release:
            abs_code_path = os.path.join(package_path, self.code_path)
            tar_code_path = self.path(self.tar_code_name)
            with tarfile.open(tar_code_path, 'w:gz') as tf:
                tf.add(abs_code_path, self.code_path)
            self.create_resource(
                description='Alert syncer source code',
                resource_path=tar_code_path,
                resource_type=ALERT_SYNCER_CODE,
                arch='linux'
            )

            req_path = os.path.join(abs_code_path, self.req_file_name)
            wheels_dir_path = os.path.join(abs_code_path, self.wheels_dir_name)
            with VirtualEnvironment(self.path('env')) as venv:
                venv.pip('pip')
                venv.pip('--no-index --find-links {} -r {}'.format(wheels_dir_path, req_path))
                venv.pack_bundle(self.tar_venv_name)
            self.create_resource(
                description='Virtual env',
                resource_path=self.path(self.tar_venv_name),
                resource_type=ALERT_SYNCER_VIRTUAL_ENV,
                arch='linux'
            )


__Task__ = BuildNannyAlertSyncer
