import time
import datetime
import logging

from sandbox.common.types.client import Tag
from sandbox import sdk2
from sandbox.common import rest
from sandbox.common import proxy
from sandbox.projects.common.nanny import nanny


RESOURCE_READY_STATE = 'READY'


class Service(object):
    def __init__(self, service_id, sandbox_resource_ids):
        self.id = service_id
        self.sandbox_resource_ids = sandbox_resource_ids

    @staticmethod
    def is_valid_resource_id(r_id):
        # Id may be equal to '', handle this case separately
        if not r_id:
            return False
        # Id may be equal to '0' or '000' or '-1'. If we try to touch such resource ids,
        # Sandbox response will be:
        # {"error":"Bad Request","reason":"Error parsing id: `id[]` must be from 1 to None"}
        # So we have to avoid such ids.
        try:
            r_id_int = int(r_id)
            # r_id_int > 0 is enough, but afraid of deleting working code.
            return r_id_int > 0 and bool(r_id_int)
        except ValueError:
            return False

    @staticmethod
    def from_api(d):
        ids = set()
        for f in d['runtime_attrs']['content']['resources']['sandbox_files']:
            rid = f.get('resource_id')
            if Service.is_valid_resource_id(rid):
                ids.add(rid)

        instance_spec = d['runtime_attrs']['content'].get('instance_spec', {})
        os_spec = instance_spec.get('osContainerSpec', {})

        for spec in (instance_spec, os_spec):
            for l in spec.get('layersConfig', {}).get('layer', []):
                m = l.get('fetchableMeta', {})
                if m.get('type') != 'SANDBOX_RESOURCE':
                    continue
                rid = m.get('sandboxResource', {}).get('resourceId')
                if Service.is_valid_resource_id(rid):
                    ids.add(rid)

        return Service(service_id=d['_id'], sandbox_resource_ids=ids)

    @staticmethod
    def from_api_list(l):
        rv = []
        for d in l.get('result', []):
            rv.append(Service.from_api(d))
        return rv


class SandboxResource(object):
    def __init__(self, resource_id, state, expired_dt):
        self.id = resource_id
        self.state = state
        self.expired_dt = expired_dt

    @staticmethod
    def from_api(d):
        s = d['time'].get('expires')
        if not s:
            expired_dt = None
        else:
            expired_dt = SandboxResource.parse_sandbox_timestamp(s)
            if not expired_dt:
                logging.warning("Can't parse expires date %s "
                                "in resource %s, skip it", s, d['id'])

        return SandboxResource(resource_id=d['id'],
                               state=d['state'],
                               expired_dt=expired_dt)

    @staticmethod
    def from_api_list(l):
        rv = []
        for d in l.get('items', []):
            rv.append(SandboxResource.from_api(d))
        return rv

    @staticmethod
    def parse_sandbox_timestamp(s):
        for fmt in ('%Y-%m-%dT%H:%M:%S.%fZ', '%Y-%m-%dT%H:%M:%SZ'):
            try:
                return datetime.datetime.strptime(s, fmt)
            except ValueError:
                pass

    def need_touch(self, now, delta):
        if self.state != RESOURCE_READY_STATE or not self.expired_dt:
            return

        return self.expired_dt - now <= delta


def resource_ids_from_services(services):
    ids = set()
    for s in services:
        ids.update(s.sandbox_resource_ids)
    return ids


def chunks(l, n):
    for i in xrange(0, len(l), n):
        yield l[i:i + n]


class TouchSandboxResources(sdk2.Task):

    SECRETS_OWNER = 'NANNY'
    NANNY_API_URL = 'https://nanny.yandex-team.ru'
    SANDBOX_API_URL = 'https://sandbox.yandex-team.ru/api/v1.0'

    FETCH_LIMIT = 100
    SANDBOX_FETCH_LIMIT = 30
    UPDATE_LIMIT = 5
    DELAY_SECONDS = 1

    class Requirements(sdk2.Task.Requirements):
        client_tags = Tag.LINUX_PRECISE
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        service_id = sdk2.parameters.String("Service ID", required=False)
        expired_delta_days = sdk2.parameters.Integer("Expired delta days", default=1)

    def get_nanny_client(self):
        token = sdk2.Vault.data(self.SECRETS_OWNER, 'nanny_robot_oauth_token')
        return nanny.NannyClient(api_url=self.NANNY_API_URL, oauth_token=token)

    def get_sandbox_client(self):
        token = sdk2.Vault.data(self.SECRETS_OWNER, 'sandbox_nanny_robot_oauth_token')
        return rest.Client(base_url=self.SANDBOX_API_URL, auth=proxy.OAuth(token))

    def list_resource_ids(self, nanny_client):
        ids = set()
        skip = 0
        while True:
            r = nanny_client.list_services(skip=skip, limit=self.FETCH_LIMIT)
            services = Service.from_api_list(r)
            if not services:
                break

            ids.update(resource_ids_from_services(services))
            skip += self.FETCH_LIMIT
            logging.info("Fetching services (%s), resources (%s)", skip, len(ids))
            time.sleep(self.DELAY_SECONDS)
        return ids

    def list_resource_ids_by_service_id(self, service_id, nanny_client):
        r = nanny_client.get_service(service_id)
        s = Service.from_api(r)
        return resource_ids_from_services([s])

    def list_resource_ids_to_update(self, resource_ids, now, delta,
                                    sandbox_client):
        ids = set()
        g = chunks(list(resource_ids), self.SANDBOX_FETCH_LIMIT)
        for n, c in enumerate(g):
            resp = sandbox_client.resource.read(id=c,
                                                state=RESOURCE_READY_STATE,
                                                limit=len(c))
            resources = SandboxResource.from_api_list(resp)
            for r in resources:
                if r.need_touch(now, delta):
                    ids.add(r.id)

            logging.info("Fetching resources (%s)", n * self.SANDBOX_FETCH_LIMIT)
            time.sleep(self.DELAY_SECONDS)

        logging.info("Resources to touch %s", len(ids))
        return ids

    def on_execute(self):
        nanny_client = self.get_nanny_client()
        sandbox_client = self.get_sandbox_client()
        service_id = self.Parameters.service_id
        now = datetime.datetime.utcnow()
        delta = datetime.timedelta(days=self.Parameters.expired_delta_days)

        if service_id:
            ids_to_fetch = self.list_resource_ids_by_service_id(service_id,
                                                                nanny_client)
        else:
            ids_to_fetch = self.list_resource_ids(nanny_client)
        ids_to_update = self.list_resource_ids_to_update(ids_to_fetch,
                                                         now,
                                                         delta,
                                                         sandbox_client)

        for c in chunks(list(ids_to_update), self.UPDATE_LIMIT):
            sandbox_client.batch.resources.touch.update(c)
            logging.info("Successfully touched resources %s", c)
            time.sleep(self.DELAY_SECONDS)
