# coding: utf-8
from __future__ import absolute_import, print_function

from datetime import datetime
import logging
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import process as sdk2_process
from sandbox.sandboxsdk import process

from sandbox.projects.common.juggler.jclient import send_events_to_juggler
import sandbox.common.types.client as ctc


class NetmonTopology(sdk2.Resource):
    """
        Resource with hosts data in topology.msgpack.gz
    """
    mds_http = sdk2.parameters.String()
    mds_https = sdk2.parameters.String()


class BuildNetmonTopology(sdk2.Task):
    """
        Download hosts data from gencfg and update it with netmap data from racktables.
    """

    s3_bucket_name = 'netmon'
    s3_endpoint = 's3.mds.yandex.net'
    max_objects = 100

    class Requirements(sdk2.Task.Requirements):
        disk_space = 2048
        ram = 2048
        client_tags = (ctc.Tag.LINUX_PRECISE | ctc.Tag.CUSTOM_JUGGLER)

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600
        build_topology_resource = sdk2.parameters.Resource("build_topology resource")
        staff_secret = sdk2.parameters.YavSecret(
            'Yav secret with Staff OAuth token',
            default='sec-01djq6sgg1ndj0x0dyq3vtg6sm@ver-01djq6sgga7hnbgjdckfxfnsdw'
        )
        s3_secret = sdk2.parameters.YavSecret(
            'Yav secret with S3 access key',
            default='sec-01e4nvhhqc7tgjdfmwbtr6yept@ver-01e4nvhhqwj9bjs3hev6j03kqa'
        )

    def _upload_to_s3(self, file_name):
        import boto3

        s3_secret = self.Parameters.s3_secret.data()
        session = boto3.session.Session(
            aws_access_key_id=s3_secret['aws_access_key_id'],
            aws_secret_access_key=s3_secret['aws_secret_access_key']
        )
        s3 = session.resource(
            service_name='s3',
            endpoint_url='https://' + self.s3_endpoint
        )
        bucket = s3.Bucket(self.s3_bucket_name)

        object_name = datetime.now().strftime('topology_%Y_%m_%d__%H_%M_%S')
        bucket.upload_file(file_name, object_name)
        object_url = '{}.{}/{}'.format(self.s3_bucket_name, self.s3_endpoint, object_name)

        try:
            objects = [obj for obj in bucket.objects.filter(Prefix='topology')]
            objects.sort(key=lambda x: x.last_modified)
            while len(objects) > self.max_objects:
                logging.info("deleting object: %r", objects[0])
                objects[0].delete()
                del objects[0]
        except Exception:
            logging.exception("Can't clean S3 bucket")

        return object_url

    def on_execute(self):
        resource = self.Parameters.build_topology_resource
        resource_data = sdk2.ResourceData(resource)  # synchronizing resource data on disk
        resource_path = str(resource_data.path)

        staff_token = self.Parameters.staff_secret.data()['robot-netbeard-sandbox-api-oauth']
        topology_path = str(self.path('topology.msgpack.gz'))
        with process.CustomOsEnviron({"STAFF_OAUTH_TOKEN": staff_token}):
            with sdk2.helpers.ProcessLog(self, logger="env") as pl:
                pl.logger.propagate = 1
                ret = sdk2_process.subprocess.Popen(
                    [os.path.join(resource_path, 'build_topology', 'build_topology'), '--out', topology_path],
                    stderr=sdk2_process.subprocess.STDOUT
                ).wait()
        assert ret == 0, 'expected return code 0, got {}'.format(ret)

        s3_object_url = self._upload_to_s3(topology_path)

        output_resource = NetmonTopology(self, "Netmon Topology", topology_path)
        output_resource.mds_http = 'http://' + s3_object_url
        output_resource.mds_https = 'https://' + s3_object_url
        sdk2.ResourceData(output_resource).ready()

        send_events_to_juggler('juggler.tasks', 'build_netmon_topology', 'OK', 'Passed')
