import os
import tarfile
import sandbox.common.types.misc as ctm

from sandbox.projects.common.nanny import nanny
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.git import Git
from sandbox.sdk2.path import Path

from sandbox.sandboxsdk import environments

from sandbox.projects.netmon.resource_types import NETMON_UI


class BuildNetmonUI(nanny.ReleaseToNannyTask2, sdk2.Task):
    work_dir = 'netmon'
    ui_tarname = 'netmon.tar.gz'

    class Requirements(sdk2.Requirements):
        environments = (
            environments.NodeJS('10.14.2'),
            environments.GCCEnvironment(version="5.3.0"),
        )

        disk_space = 1024 * 5
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Git parameters") as git_block:
            repository = sdk2.parameters.String("Repository", required=True, default="ssh://git@bb.yandex-team.ru/infracloudui/netmon.git")
            ref_id = sdk2.parameters.String("Branch", default="master")
            ref_sha = sdk2.parameters.String("Commit", default=None)

        with sdk2.parameters.Group("Vault") as vault_block:
            ssh_vault_name = sdk2.parameters.String("Vault item with ssh key for git access", default="env.SSH_KEY")
            ssh_vault_owner = sdk2.parameters.String("Vault item owner", default="INFRACLOUDUI")

    def git_clone(self):
        with sdk2.ssh.Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
            git = Git(self.Parameters.repository)
            git.clone(self.work_dir, self.Parameters.ref_id, self.Parameters.ref_sha)

    def build_ui(self):
        ui_path = str(self.path(self.work_dir))
        dist_path = os.path.join(ui_path, 'build')
        tar_path = self.path(self.ui_tarname)
        tag = self.Parameters.ref_id

        if tag.startswith('refs/heads/release/'):
            os.environ['CI'] = 'true'
            os.environ['SENTRY_AUTH_TOKEN'] = sdk2.Vault.data('INFRACLOUDUI', 'env.SENTRY_AUTH_TOKEN')

        with sdk2.helpers.ProcessLog(self, logger='build_ui') as pl:
            sp.check_call(
                ['npm', 'install'],
                cwd=ui_path,
                stdout=pl.stdout,
                stderr=pl.stderr
            )
            sp.check_call(
                ['npm', 'run', 'build'],
                cwd=ui_path,
                stdout=pl.stdout,
                stderr=pl.stderr
            )

        with tarfile.open(self.ui_tarname, 'w:gz') as tar:
            tar.add(dist_path, arcname=os.path.basename(dist_path))

        resource = sdk2.ResourceData(NETMON_UI(
            self,
            'NETMON UI, tag {}'.format(self.Parameters.ref_id),
            self.ui_tarname
        ))

        resource.path.write_bytes(Path(tar_path).read_bytes())

    def on_execute(self):
        self.git_clone()
        self.build_ui()
