import requests
from sandbox import sdk2
from sandbox.projects.common import task_env


class NetmonCloudTopology(sdk2.Resource):
    """
        Resource with YaCloud hosts data in topology.msgpack.gz
    """
    mds_http = sdk2.parameters.String()
    mds_https = sdk2.parameters.String()


class NetmonCloudPreprodTopology(sdk2.Resource):
    """
        Resource with YaCloud preprod hosts data in topology.msgpack.gz
    """
    mds_http = sdk2.parameters.String()
    mds_https = sdk2.parameters.String()


class UpdateNetmonCloudTopology(sdk2.Task):

    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        topology_url = sdk2.parameters.String(
            'Topology URL',
            default='netinfra-netmon.s3.mds.yandex.net/prod/topology.msgpack.gz'
        )
        resource_type = sdk2.parameters.String(
            'Output resource type (NETMON_CLOUD_TOPOLOGY / NETMON_CLOUD_PREPROD_TOPOLOGY)',
            default='NETMON_CLOUD_TOPOLOGY'
        )

    def on_execute(self):
        url = self.Parameters.topology_url
        if url.startswith('http://'):
            url = url[len('http://'):]
        elif url.startswith('https://'):
            url = url[len('https://'):]

        response = requests.get('http://' + url)
        response.raise_for_status()

        if self.Parameters.resource_type == 'NETMON_CLOUD_TOPOLOGY':
            output_resource = NetmonCloudTopology(
                self, 'Netmon cloud topology', 'topology.msgpack.gz')
        elif self.Parameters.resource_type == 'NETMON_CLOUD_PREPROD_TOPOLOGY':
            output_resource = NetmonCloudPreprodTopology(
                self, 'Netmon cloud preprod topology', 'topology.msgpack.gz')
        else:
            raise RuntimeError('Unexpected resource type "{}"'.format(self.Parameters.resource_type))

        output_resource.mds_http = 'http://' + url
        output_resource.mds_https = 'https://' + url
        data = sdk2.ResourceData(output_resource)
        data.path.write_bytes(response.content)
        data.ready()
